/*
 * Copyright (C) 2017 Spreadtrum Communications Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/pm_wakeirq.h>
#include <linux/regmap.h>
#include <linux/rtc.h>
#include <linux/sched.h>

#define SPRD_RTC_SEC_CNT_VALUE		0x0
#define SPRD_RTC_MIN_CNT_VALUE		0x4
#define SPRD_RTC_HOUR_CNT_VALUE		0x8
#define SPRD_RTC_DAY_CNT_VALUE		0xc
#define SPRD_RTC_SEC_CNT_UPD		0x10
#define SPRD_RTC_MIN_CNT_UPD		0x14
#define SPRD_RTC_HOUR_CNT_UPD		0x18
#define SPRD_RTC_DAY_CNT_UPD		0x1c
#define SPRD_RTC_SEC_ALM_UPD		0x20
#define SPRD_RTC_MIN_ALM_UPD		0x24
#define SPRD_RTC_HOUR_ALM_UPD		0x28
#define SPRD_RTC_DAY_ALM_UPD		0x2c
#define SPRD_RTC_INT_EN			0x30
#define SPRD_RTC_INT_RAW_STS		0x34
#define SPRD_RTC_INT_CLR		0x38
#define SPRD_RTC_INT_MASK_STS		0x3C
#define SPRD_RTC_SEC_ALM_VALUE		0x40
#define SPRD_RTC_MIN_ALM_VALUE		0x44
#define SPRD_RTC_HOUR_ALM_VALUE		0x48
#define SPRD_RTC_DAY_ALM_VALUE		0x4c
#define SPRD_RTC_SPG_VALUE		0x50
#define SPRD_RTC_SPG_UPD		0x54
#define SPRD_RTC_PWR_CTRL		0x58
#define SPRD_RTC_PWR_STS		0x5c
#define SPRD_RTC_SEC_AUXALM_UPD		0x60
#define SPRD_RTC_MIN_AUXALM_UPD		0x64
#define SPRD_RTC_HOUR_AUXALM_UPD	0x68
#define SPRD_RTC_DAY_AUXALM_UPD		0x6c

/* BIT & MASK definition for SPRD_RTC_INT_* registers */
#define SPRD_RTC_SEC_EN			BIT(0)
#define SPRD_RTC_MIN_EN			BIT(1)
#define SPRD_RTC_HOUR_EN		BIT(2)
#define SPRD_RTC_DAY_EN			BIT(3)
#define SPRD_RTC_ALARM_EN		BIT(4)
#define SPRD_RTC_HRS_FORMAT_EN		BIT(5)
#define SPRD_RTC_AUXALM_EN		BIT(6)
#define SPRD_RTC_SPG_UPD_EN		BIT(7)
#define SPRD_RTC_SEC_UPD_EN		BIT(8)
#define SPRD_RTC_MIN_UPD_EN		BIT(9)
#define SPRD_RTC_HOUR_UPD_EN		BIT(10)
#define SPRD_RTC_DAY_UPD_EN		BIT(11)
#define SPRD_RTC_ALMSEC_UPD_EN		BIT(12)
#define SPRD_RTC_ALMMIN_UPD_EN		BIT(13)
#define SPRD_RTC_ALMHOUR_UPD_EN		BIT(14)
#define SPRD_RTC_ALMDAY_UPD_EN		BIT(15)
#define SPRD_RTC_INT_MASK		GENMASK(15, 0)

#define SPRD_RTC_TIME_INT_MASK				\
	(SPRD_RTC_SEC_UPD_EN | SPRD_RTC_MIN_UPD_EN |	\
	 SPRD_RTC_HOUR_UPD_EN | SPRD_RTC_DAY_UPD_EN)

#define SPRD_RTC_ALMTIME_INT_MASK				\
	(SPRD_RTC_ALMSEC_UPD_EN | SPRD_RTC_ALMMIN_UPD_EN |	\
	 SPRD_RTC_ALMHOUR_UPD_EN | SPRD_RTC_ALMDAY_UPD_EN)

#define SPRD_RTC_ALM_INT_MASK			\
	(SPRD_RTC_SEC_EN | SPRD_RTC_MIN_EN |	\
	 SPRD_RTC_HOUR_EN | SPRD_RTC_DAY_EN |	\
	 SPRD_RTC_ALARM_EN | SPRD_RTC_AUXALM_EN)

/* second/minute/hour/day values mask definition */
#define SPRD_RTC_SEC_MASK		GENMASK(5, 0)
#define SPRD_RTC_MIN_MASK		GENMASK(5, 0)
#define SPRD_RTC_HOUR_MASK		GENMASK(4, 0)
#define SPRD_RTC_DAY_MASK		GENMASK(15, 0)

/* alarm lock definition for SPRD_RTC_SPG_UPD register */
#define SPRD_RTC_ALMLOCK_MASK		GENMASK(7, 0)
#define SPRD_RTC_ALM_UNLOCK		0xa5
#define SPRD_RTC_ALM_LOCK		(~SPRD_RTC_ALM_UNLOCK &	\
					 SPRD_RTC_ALMLOCK_MASK)

/* SPG values definition for SPRD_RTC_SPG_UPD register */
#define SPRD_RTC_POWEROFF_ALM_FLAG	BIT(8)

/* power control/status definition */
#define SPRD_RTC_POWER_RESET_VALUE	0x96
#define SPRD_RTC_POWER_STS_CLEAR	GENMASK(7, 0)
#define SPRD_RTC_POWER_STS_SHIFT	8
#define SPRD_RTC_POWER_STS_VALID	\
	(~SPRD_RTC_POWER_RESET_VALUE << SPRD_RTC_POWER_STS_SHIFT)

/* timeout of synchronizing time and alarm registers (us) */
#define SPRD_RTC_POLL_TIMEOUT		200000
#define SPRD_RTC_POLL_DELAY_US		20000

/* default size of uevent trace log buffer */
#define SPRD_RTC_UEVENT_LOG_SIZE	256

static int sprd_rtcdbg_log_force;
module_param(sprd_rtcdbg_log_force, int, 0644);
MODULE_PARM_DESC(sprd_rtcdbg_log_force, "sprd rtcdbg log force out (default: 0)");
#define SPRD_RTCDBG_INFO(fmt, ...)							\
	do {										\
		if (!sprd_rtcdbg_log_force)						\
			pr_info("[%s] "pr_fmt(fmt), "SPRD_RTCDBG", ##__VA_ARGS__);	\
		else {									\
			pr_err("[%s] "pr_fmt(fmt), "SPRD_RTCDBG", ##__VA_ARGS__);	\
			dump_stack();							\
		}									\
	} while (0)									\

#define SPRD_RTCDBG_ERROR(fmt, ...)							\
		pr_err("[%s] "pr_fmt(fmt), "SPRD_RTCERR", ##__VA_ARGS__)

struct sprd_rtc {
	struct rtc_device	*rtc;
	struct regmap		*regmap;
	struct device		*dev;
	u32			base;
	int			irq;
	bool			valid;
	bool			registered;
	char			alrm_comm[128];
};

/*
 * The Spreadtrum RTC controller has 3 groups registers, including time, normal
 * alarm and auxiliary alarm. The time group registers are used to set RTC time,
 * the normal alarm registers are used to set normal alarm, and the auxiliary
 * alarm registers are used to set auxiliary alarm. Both alarm event and
 * auxiliary alarm event can wake up system from deep sleep, but only alarm
 * event can power up system from power down status.
 */
enum sprd_rtc_reg_types {
	SPRD_RTC_TIME,
	SPRD_RTC_ALARM,
	SPRD_RTC_AUX_ALARM,
};

static int sprd_rtc_uevent_notify(struct device *dev, char *event_str)
{
	char *uevent = NULL;
	char *pr_str[2] = { uevent, NULL };

	uevent = devm_kmalloc(dev, 2 * SPRD_RTC_UEVENT_LOG_SIZE, GFP_KERNEL);
	if (uevent != NULL) {
		snprintf(uevent, 2 * SPRD_RTC_UEVENT_LOG_SIZE,
			 "kevent_begin:{\"event_id\":\"107000004\",\"event_time\":%lld,%s}:kevent_end",
			 ktime_to_ms(ktime_get_boottime()), event_str);
		kobject_uevent_env(&(dev->kobj), KOBJ_CHANGE, pr_str);
		devm_kfree(dev, uevent);
	} else {
		SPRD_RTCDBG_ERROR("failed to allocate memory for uevent logging");
		return -ENOMEM;
	}

	return 0;
}

static int sprd_rtc_clear_alarm_ints(struct sprd_rtc *rtc)
{
	return regmap_write(rtc->regmap, rtc->base + SPRD_RTC_INT_CLR,
			    SPRD_RTC_ALM_INT_MASK);
}

static int sprd_rtc_lock_alarm(struct sprd_rtc *rtc, bool lock)
{
	int ret;
	u32 val;
	u32 spg_val;

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_SPG_VALUE, &val);
	if (ret)
		return ret;

	SPRD_RTCDBG_INFO("set alarm lock: %u, process pid: %u, name: %s\n",
			 (u32)lock, current->pid, current->comm);

	spg_val = val & SPRD_RTC_ALMLOCK_MASK;
	if ((lock && (spg_val == SPRD_RTC_ALM_LOCK)) ||
		(!lock && (spg_val == SPRD_RTC_ALM_UNLOCK))) {
		SPRD_RTCDBG_INFO("alarm lock keep up with the prev\n");
		return ret;
	}

	val &= ~SPRD_RTC_ALMLOCK_MASK;
	if (lock)
		val |= SPRD_RTC_ALM_LOCK;
	else
		val |= SPRD_RTC_ALM_UNLOCK | SPRD_RTC_POWEROFF_ALM_FLAG;

	ret = regmap_write(rtc->regmap, rtc->base + SPRD_RTC_INT_CLR,
			   SPRD_RTC_SPG_UPD_EN);
	if (ret)
		return ret;

	ret = regmap_write(rtc->regmap, rtc->base + SPRD_RTC_SPG_UPD, val);
	if (ret)
		return ret;

	/*
	 * It takes too long to resume alarmtimer device，about 200ms, which
	 * affects the system resume time. The reason is that the system would
	 * lock alarm if there are not alarms in the timerqueue, and the sprd
	 * chip spec claims that requires about 125ms to take effect when set
	 * rtc register to lock alarm on the chip. In order to optimize system
	 * resuming time, we delay 5~6ms to ensure the lock info is set to the
	 * chip instead of waiting the register is updated successfully.
	 * System would unlock alarm when shutdown the device and there is a
	 * poweroff alarm, so we should wait until the register is updated
	 * successfully before system shutdown.
	 */

	ret = regmap_read_poll_timeout(rtc->regmap,
				       rtc->base + SPRD_RTC_INT_RAW_STS, val,
				       (val & SPRD_RTC_SPG_UPD_EN),
				       SPRD_RTC_POLL_DELAY_US,
				       SPRD_RTC_POLL_TIMEOUT);
	if (ret)
		dev_err(rtc->dev, "failed to update SPG value:%d\n", ret);

	return ret;
}

static int sprd_rtc_get_secs(struct sprd_rtc *rtc, enum sprd_rtc_reg_types type,
			     time64_t *secs)
{
	u32 sec_reg, min_reg, hour_reg, day_reg;
	u32 val, sec, min, hour, day;
	int ret;

	switch (type) {
	case SPRD_RTC_TIME:
		sec_reg = SPRD_RTC_SEC_CNT_VALUE;
		min_reg = SPRD_RTC_MIN_CNT_VALUE;
		hour_reg = SPRD_RTC_HOUR_CNT_VALUE;
		day_reg = SPRD_RTC_DAY_CNT_VALUE;
		break;
	case SPRD_RTC_ALARM:
		sec_reg = SPRD_RTC_SEC_ALM_VALUE;
		min_reg = SPRD_RTC_MIN_ALM_VALUE;
		hour_reg = SPRD_RTC_HOUR_ALM_VALUE;
		day_reg = SPRD_RTC_DAY_ALM_VALUE;
		break;
	case SPRD_RTC_AUX_ALARM:
		sec_reg = SPRD_RTC_SEC_AUXALM_UPD;
		min_reg = SPRD_RTC_MIN_AUXALM_UPD;
		hour_reg = SPRD_RTC_HOUR_AUXALM_UPD;
		day_reg = SPRD_RTC_DAY_AUXALM_UPD;
		break;
	default:
		return -EINVAL;
	}

	ret = regmap_read(rtc->regmap, rtc->base + sec_reg, &val);
	if (ret)
		return ret;

	sec = val & SPRD_RTC_SEC_MASK;

	ret = regmap_read(rtc->regmap, rtc->base + min_reg, &val);
	if (ret)
		return ret;

	min = val & SPRD_RTC_MIN_MASK;

	ret = regmap_read(rtc->regmap, rtc->base + hour_reg, &val);
	if (ret)
		return ret;

	hour = val & SPRD_RTC_HOUR_MASK;

	ret = regmap_read(rtc->regmap, rtc->base + day_reg, &val);
	if (ret)
		return ret;

	day = val & SPRD_RTC_DAY_MASK;
	*secs = (((time64_t)(day * 24) + hour) * 60 + min) * 60 + sec;
	return 0;
}

static int sprd_rtc_set_secs(struct sprd_rtc *rtc, enum sprd_rtc_reg_types type,
			     time64_t secs)
{
	u32 sec_reg, min_reg, hour_reg, day_reg, sts_mask;
	u32 sec, min, hour, day, val;
	int ret, rem;

	/* convert seconds to RTC time format */
	day = div_s64_rem(secs, 86400, &rem);
	hour = rem / 3600;
	rem -= hour * 3600;
	min = rem / 60;
	sec = rem - min * 60;

	switch (type) {
	case SPRD_RTC_TIME:
		sec_reg = SPRD_RTC_SEC_CNT_UPD;
		min_reg = SPRD_RTC_MIN_CNT_UPD;
		hour_reg = SPRD_RTC_HOUR_CNT_UPD;
		day_reg = SPRD_RTC_DAY_CNT_UPD;
		sts_mask = SPRD_RTC_TIME_INT_MASK;
		break;
	case SPRD_RTC_ALARM:
		sec_reg = SPRD_RTC_SEC_ALM_UPD;
		min_reg = SPRD_RTC_MIN_ALM_UPD;
		hour_reg = SPRD_RTC_HOUR_ALM_UPD;
		day_reg = SPRD_RTC_DAY_ALM_UPD;
		sts_mask = SPRD_RTC_ALMTIME_INT_MASK;
		break;
	case SPRD_RTC_AUX_ALARM:
		sec_reg = SPRD_RTC_SEC_AUXALM_UPD;
		min_reg = SPRD_RTC_MIN_AUXALM_UPD;
		hour_reg = SPRD_RTC_HOUR_AUXALM_UPD;
		day_reg = SPRD_RTC_DAY_AUXALM_UPD;
		sts_mask = 0;
		break;
	default:
		return -EINVAL;
	}

	ret = regmap_write(rtc->regmap, rtc->base + sec_reg, sec);
	if (ret)
		return ret;

	ret = regmap_write(rtc->regmap, rtc->base + min_reg, min);
	if (ret)
		return ret;

	ret = regmap_write(rtc->regmap, rtc->base + hour_reg, hour);
	if (ret)
		return ret;

	ret = regmap_write(rtc->regmap, rtc->base + day_reg, day);
	if (ret)
		return ret;

	if (type == SPRD_RTC_AUX_ALARM)
		return 0;

	/*
	 * Since the time and normal alarm registers are put in always-power-on
	 * region supplied by VDDRTC, then these registers changing time will
	 * be very long, about 125ms. Thus here we should wait until all
	 * values are updated successfully.
	 */
	ret = regmap_read_poll_timeout(rtc->regmap,
				       rtc->base + SPRD_RTC_INT_RAW_STS, val,
				       ((val & sts_mask) == sts_mask),
				       SPRD_RTC_POLL_DELAY_US,
				       SPRD_RTC_POLL_TIMEOUT);
	if (ret < 0) {
		dev_err(rtc->dev, "set time/alarm values timeout\n");
		return ret;
	}

	return regmap_write(rtc->regmap, rtc->base + SPRD_RTC_INT_CLR,
			    sts_mask);
}

static int sprd_rtc_read_aux_alarm(struct device *dev, struct rtc_wkalrm *alrm)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	time64_t secs;
	u32 val;
	int ret;

	ret = sprd_rtc_get_secs(rtc, SPRD_RTC_AUX_ALARM, &secs);
	if (ret)
		return ret;

	rtc_time64_to_tm(secs, &alrm->time);

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_INT_EN, &val);
	if (ret)
		return ret;

	alrm->enabled = !!(val & SPRD_RTC_AUXALM_EN);

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_INT_RAW_STS, &val);
	if (ret)
		return ret;

	alrm->pending = !!(val & SPRD_RTC_AUXALM_EN);
	return 0;
}

static int sprd_rtc_set_aux_alarm(struct device *dev, struct rtc_wkalrm *alrm)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	time64_t secs = rtc_tm_to_time64(&alrm->time);
	int ret;

	SPRD_RTCDBG_INFO("setting aux_alarm: %d-%d-%d %d:%d:%d\n", alrm->time.tm_year + 1900,
		       alrm->time.tm_mon + 1, alrm->time.tm_mday, alrm->time.tm_hour,
		       alrm->time.tm_min, alrm->time.tm_sec);

	/* clear the auxiliary alarm interrupt status */
	ret = regmap_write(rtc->regmap, rtc->base + SPRD_RTC_INT_CLR,
			   SPRD_RTC_AUXALM_EN);
	if (ret)
		return ret;

	ret = sprd_rtc_set_secs(rtc, SPRD_RTC_AUX_ALARM, secs);
	if (ret)
		return ret;

	if (alrm->enabled) {
		ret = regmap_update_bits(rtc->regmap,
					 rtc->base + SPRD_RTC_INT_EN,
					 SPRD_RTC_AUXALM_EN,
					 SPRD_RTC_AUXALM_EN);
	} else {
		ret = regmap_update_bits(rtc->regmap,
					 rtc->base + SPRD_RTC_INT_EN,
					 SPRD_RTC_AUXALM_EN, 0);
	}

	return ret;
}

static int sprd_rtc_read_time(struct device *dev, struct rtc_time *tm)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	time64_t secs;
	int ret;

	if (!rtc->valid) {
		dev_warn(dev, "RTC values are invalid\n");
		return -EINVAL;
	}

	ret = sprd_rtc_get_secs(rtc, SPRD_RTC_TIME, &secs);
	if (ret)
		return ret;

	rtc_time64_to_tm(secs, tm);
	return 0;
}

static int sprd_rtc_set_time(struct device *dev, struct rtc_time *tm)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	time64_t secs = rtc_tm_to_time64(tm);
	int ret;

	SPRD_RTCDBG_INFO("setting time: %d-%d-%d %d:%d:%d\n", tm->tm_year + 1900,
			 tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min,
			 tm->tm_sec);

	ret = sprd_rtc_set_secs(rtc, SPRD_RTC_TIME, secs);
	if (ret)
		return ret;

	if (!rtc->valid) {
		/* Clear RTC power status firstly */
		ret = regmap_write(rtc->regmap, rtc->base + SPRD_RTC_PWR_CTRL,
				   SPRD_RTC_POWER_STS_CLEAR);
		if (ret)
			return ret;

		/*
		 * Set RTC power status to indicate now RTC has valid time
		 * values.
		 */
		ret = regmap_write(rtc->regmap, rtc->base + SPRD_RTC_PWR_CTRL,
				   SPRD_RTC_POWER_STS_VALID);
		if (ret)
			return ret;

		rtc->valid = true;
	}

	return 0;
}

static int sprd_rtc_read_alarm(struct device *dev, struct rtc_wkalrm *alrm)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	time64_t secs;
	int ret;
	u32 val;

	/*
	 * Before RTC device is registered, it will check to see if there is an
	 * alarm already set in RTC hardware, and we always read the normal
	 * alarm at this time.
	 *
	 * Or if aie_timer is enabled, we should get the normal alarm time.
	 * Otherwise we should get auxiliary alarm time.
	 */
	if (rtc->rtc && rtc->registered && rtc->rtc->aie_timer.enabled == 0)
		return sprd_rtc_read_aux_alarm(dev, alrm);

	ret = sprd_rtc_get_secs(rtc, SPRD_RTC_ALARM, &secs);
	if (ret)
		return ret;

	rtc_time64_to_tm(secs, &alrm->time);

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_INT_EN, &val);
	if (ret)
		return ret;

	alrm->enabled = !!(val & SPRD_RTC_ALARM_EN);

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_INT_RAW_STS, &val);
	if (ret)
		return ret;

	alrm->pending = !!(val & SPRD_RTC_ALARM_EN);
	return 0;
}

static int sprd_rtc_set_alarm(struct device *dev, struct rtc_wkalrm *alrm)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	time64_t secs_alarm;
	time64_t secs = rtc_tm_to_time64(&alrm->time);
	struct rtc_time aie_time =
		rtc_ktime_to_tm(rtc->rtc->aie_timer.node.expires);
	int ret;

	strlcpy(rtc->alrm_comm, current->comm, sizeof(rtc->alrm_comm));

	/*
	 * We have 2 groups alarms: normal alarm and auxiliary alarm. Since
	 * both normal alarm event and auxiliary alarm event can wake up system
	 * from deep sleep, but only alarm event can power up system from power
	 * down status. Moreover we do not need to poll about 125ms when
	 * updating auxiliary alarm registers. Thus we usually set auxiliary
	 * alarm when wake up system from deep sleep, and for other scenarios,
	 * we should set normal alarm with polling status.
	 *
	 * So here we check if the alarm time is set by aie_timer, if yes, we
	 * should set normal alarm, if not, we should set auxiliary alarm which
	 * means it is just a wake event.
	 */
	if (!rtc->rtc->aie_timer.enabled || rtc_tm_sub(&aie_time, &alrm->time))
		return sprd_rtc_set_aux_alarm(dev, alrm);

	SPRD_RTCDBG_INFO("setting normal_alarm: %d-%d-%d %d:%d:%d\n", alrm->time.tm_year + 1900,
		       alrm->time.tm_mon + 1, alrm->time.tm_mday, alrm->time.tm_hour,
		       alrm->time.tm_min, alrm->time.tm_sec);

	/* clear the alarm interrupt status firstly */
	ret = regmap_write(rtc->regmap, rtc->base + SPRD_RTC_INT_CLR,
			   SPRD_RTC_ALARM_EN);
	if (ret)
		return ret;

	ret = sprd_rtc_get_secs(rtc, SPRD_RTC_ALARM, &secs_alarm);
	if (ret)
		return ret;

	if (secs_alarm != secs) {
		ret = sprd_rtc_set_secs(rtc, SPRD_RTC_ALARM, secs);
		if (ret)
			return ret;
	}

	if (alrm->enabled) {
		ret = regmap_update_bits(rtc->regmap,
					 rtc->base + SPRD_RTC_INT_EN,
					 SPRD_RTC_ALARM_EN,
					 SPRD_RTC_ALARM_EN);
		if (ret)
			return ret;

		/* unlock the alarm to enable the alarm function. */
		ret = sprd_rtc_lock_alarm(rtc, false);
	} else {
		regmap_update_bits(rtc->regmap,
				   rtc->base + SPRD_RTC_INT_EN,
				   SPRD_RTC_ALARM_EN, 0);

		/*
		 * Lock the alarm function in case fake alarm event will power
		 * up systems.
		 */
		ret = sprd_rtc_lock_alarm(rtc, true);
	}

	return ret;
}

static int sprd_rtc_alarm_irq_enable(struct device *dev, unsigned int enabled)
{
	struct sprd_rtc *rtc = dev_get_drvdata(dev);
	int ret;

	if (enabled) {
		ret = regmap_update_bits(rtc->regmap,
					 rtc->base + SPRD_RTC_INT_EN,
					 SPRD_RTC_ALARM_EN | SPRD_RTC_AUXALM_EN,
					 SPRD_RTC_ALARM_EN | SPRD_RTC_AUXALM_EN);
		if (ret)
			return ret;

		ret = sprd_rtc_lock_alarm(rtc, false);
	} else {
		regmap_update_bits(rtc->regmap, rtc->base + SPRD_RTC_INT_EN,
				   SPRD_RTC_ALARM_EN | SPRD_RTC_AUXALM_EN, 0);

		ret = sprd_rtc_lock_alarm(rtc, true);
	}

	return ret;
}

static const struct rtc_class_ops sprd_rtc_ops = {
	.read_time = sprd_rtc_read_time,
	.set_time = sprd_rtc_set_time,
	.read_alarm = sprd_rtc_read_alarm,
	.set_alarm = sprd_rtc_set_alarm,
	.alarm_irq_enable = sprd_rtc_alarm_irq_enable,
};

static irqreturn_t sprd_rtc_handler(int irq, void *dev_id)
{
	struct sprd_rtc *rtc = dev_id;
	int ret;
	struct rtc_time tm;
	char *uevent_str;

	ret = sprd_rtc_clear_alarm_ints(rtc);
	if (ret)
		return IRQ_RETVAL(ret);

	ret = sprd_rtc_read_time(rtc->rtc->dev.parent, &tm);
	if (ret)
		ret = -EINVAL;

	rtc_update_irq(rtc->rtc, 1, RTC_AF | RTC_IRQF);

	SPRD_RTCDBG_INFO("now [%lld]ms, alarm set by [%s],triggered at %d-%d-%d %d:%d:%d\n",
			 ktime_to_ms(ktime_get_boottime()), rtc->alrm_comm, tm.tm_year + 1900,
			 tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);

	uevent_str = devm_kmalloc(rtc->dev, SPRD_RTC_UEVENT_LOG_SIZE, GFP_KERNEL);
	if (uevent_str != NULL) {
		snprintf(uevent_str, SPRD_RTC_UEVENT_LOG_SIZE,
			 "\"flag\":RTC_ALARM_INT,\"info\":owner:%s;alarm:%d-%d-%d %d:%d:%d",
			 rtc->alrm_comm, tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
			 tm.tm_hour, tm.tm_min, tm.tm_sec);
		sprd_rtc_uevent_notify(rtc->dev, uevent_str);
		devm_kfree(rtc->dev, uevent_str);
	} else {
		SPRD_RTCDBG_ERROR("failed to allocate memory for uevent: RTC_ALARM_INT");
	}

	return IRQ_HANDLED;
}

static int sprd_rtc_check_power_down(struct sprd_rtc *rtc)
{
	u32 val;
	int ret;
	struct rtc_time tm = {
		.tm_mday = 1,
		.tm_mon = 0,
		.tm_year = 70,
	};

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_PWR_STS, &val);
	if (ret)
		return ret;
	/*
	 * If the RTC power status value is SPRD_RTC_POWER_RESET_VALUE, which
	 * means the RTC has been powered down, so init the RTC time to
	 * 1970.0.0 0:0:0.
	 */
	if (val == SPRD_RTC_POWER_RESET_VALUE) {
		SPRD_RTCDBG_ERROR("RTC hardware has been reset, then init time\n");
		ret = sprd_rtc_set_time(rtc->dev, &tm);
		if (!ret)
			ret = sprd_rtc_set_secs(rtc, SPRD_RTC_ALARM, 0);
	} else {
		rtc->valid = true;
	}

	return ret;
}

static int sprd_rtc_check_alarm_int(struct sprd_rtc *rtc)
{
	u32 val;
	int ret, alrm_lock, pwroff_alrm;
	struct rtc_wkalrm alrm = { 0 };
	char *uevent_str;

	ret = regmap_read(rtc->regmap, rtc->base + SPRD_RTC_SPG_VALUE, &val);
	if (ret)
		return ret;

	pwroff_alrm = (val & SPRD_RTC_POWEROFF_ALM_FLAG) ? 1 : 0;
	alrm_lock = ((val & SPRD_RTC_ALMLOCK_MASK) == SPRD_RTC_ALM_UNLOCK) ? 0 : 1;
	SPRD_RTCDBG_INFO("SPG reg value: 0x%04x, alarm lock: %d, poweroff alarm flag: %d\n",
			 val, alrm_lock, pwroff_alrm);

	ret = sprd_rtc_read_alarm(rtc->dev, &alrm);
	if (ret)
		SPRD_RTCDBG_ERROR("failed to read normal alarm time\n");
	else
		SPRD_RTCDBG_INFO("alarm time value: %d-%d-%d %d:%d:%d\n", alrm.time.tm_year + 1900,
				 alrm.time.tm_mon + 1, alrm.time.tm_mday, alrm.time.tm_hour,
				 alrm.time.tm_min, alrm.time.tm_sec);

	uevent_str = devm_kmalloc(rtc->dev, SPRD_RTC_UEVENT_LOG_SIZE, GFP_KERNEL);
	if (uevent_str != NULL) {
		snprintf(uevent_str, SPRD_RTC_UEVENT_LOG_SIZE,
			 "\"flag\":RTC_CHECK_ALARM,\"info\":spg:0x%04x;alarm:%d-%d-%d %d:%d:%d",
			 val, alrm.time.tm_year + 1900, alrm.time.tm_mon + 1, alrm.time.tm_mday,
			 alrm.time.tm_hour, alrm.time.tm_min, alrm.time.tm_sec);
		sprd_rtc_uevent_notify(rtc->dev, uevent_str);
		devm_kfree(rtc->dev, uevent_str);
	} else {
		SPRD_RTCDBG_ERROR("failed to allocate memory for uevent: RTC_CHECK_ALARM");
	}

	/*
	 * The SPRD_RTC_INT_EN register is not put in always-power-on region
	 * supplied by VDDRTC, so we should check if we need enable the alarm
	 * interrupt when system booting.
	 *
	 * If we have set SPRD_RTC_POWEROFF_ALM_FLAG which is saved in
	 * always-power-on region, that means we have set one alarm last time,
	 * so we should enable the alarm interrupt to help RTC core to see if
	 * there is an alarm already set in RTC hardware.
	 */
	if (!(val & SPRD_RTC_POWEROFF_ALM_FLAG))
		return 0;

	return regmap_update_bits(rtc->regmap, rtc->base + SPRD_RTC_INT_EN,
				  SPRD_RTC_ALARM_EN, SPRD_RTC_ALARM_EN);
}

static int sprd_rtc_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct sprd_rtc *rtc;
	int ret;
	char *uevent_str;

	rtc = devm_kzalloc(&pdev->dev, sizeof(*rtc), GFP_KERNEL);
	if (!rtc)
		return -ENOMEM;

	rtc->regmap = dev_get_regmap(pdev->dev.parent, NULL);
	if (!rtc->regmap)
		return -ENODEV;

	ret = of_property_read_u32(node, "reg", &rtc->base);
	if (ret) {
		dev_err(&pdev->dev, "failed to get RTC base address\n");
		return ret;
	}

	rtc->irq = platform_get_irq(pdev, 0);
	if (rtc->irq < 0)
		return rtc->irq;

	rtc->rtc = devm_rtc_allocate_device(&pdev->dev);
	if (IS_ERR(rtc->rtc))
		return PTR_ERR(rtc->rtc);

	rtc->dev = &pdev->dev;
	platform_set_drvdata(pdev, rtc);

	/* check if RTC time values are valid */
	ret = sprd_rtc_check_power_down(rtc);
	if (ret) {
		dev_err(&pdev->dev, "failed to check RTC time values\n");
		return ret;
	}

	/* check if we need set the alarm interrupt */
	ret = sprd_rtc_check_alarm_int(rtc);
	if (ret) {
		dev_err(&pdev->dev, "failed to check RTC alarm interrupt\n");
		return ret;
	}

	device_init_wakeup(&pdev->dev, 1);
	dev_pm_set_wake_irq(&pdev->dev, rtc->irq);

	rtc->rtc->ops = &sprd_rtc_ops;
	rtc->rtc->range_min = 0;
	rtc->rtc->range_max = 5662310399LL;
	ret = devm_rtc_register_device(rtc->rtc);
	if (ret) {
		dev_err(&pdev->dev, "failed to register device to RTC class\n");
		return ret;
	}

	ret = devm_request_threaded_irq(&pdev->dev, rtc->irq, NULL,
					sprd_rtc_handler,
					IRQF_ONESHOT | IRQF_EARLY_RESUME,
					pdev->name, rtc);
	if (ret < 0) {
		dev_err(&pdev->dev, "failed to request RTC irq\n");
		return ret;
	}

	rtc->registered = true;

	uevent_str = devm_kmalloc(&pdev->dev, SPRD_RTC_UEVENT_LOG_SIZE, GFP_KERNEL);
	if (uevent_str != NULL) {
		snprintf(uevent_str, SPRD_RTC_UEVENT_LOG_SIZE,
			 "\"flag\":RTC_PROBE_DONE,\"info\":%d", rtc->registered);
		sprd_rtc_uevent_notify(&(pdev->dev), uevent_str);
		devm_kfree(&pdev->dev, uevent_str);
	} else {
		SPRD_RTCDBG_ERROR("failed to allocate memory for uevent: RTC_PROBE_DONE");
	}
	SPRD_RTCDBG_INFO("RTC driver probe done\n");

	return 0;
}

static int sprd_rtc_remove(struct platform_device *pdev)
{
	dev_pm_clear_wake_irq(&pdev->dev);
	device_init_wakeup(&pdev->dev, 0);
	return 0;
}

static const struct of_device_id sprd_rtc_of_match[] = {
	{ .compatible = "sprd,sc2731-rtc", },
	{ .compatible = "sprd,ump96xx-rtc", },
	{ },
};
MODULE_DEVICE_TABLE(of, sprd_rtc_of_match);

static struct platform_driver sprd_rtc_driver = {
	.driver = {
		.name = "sprd-rtc",
		.of_match_table = sprd_rtc_of_match,
	},
	.probe	= sprd_rtc_probe,
	.remove = sprd_rtc_remove,
};
module_platform_driver(sprd_rtc_driver);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("Spreadtrum RTC Device Driver");
MODULE_AUTHOR("Baolin Wang <baolin.wang@spreadtrum.com>");
