// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2018 Spreadtrum Communications Inc.
 */

#include <linux/debugfs.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/regmap.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/of_regulator.h>
#include <linux/sched.h>
#include <linux/delay.h>

#define UMP9620_REGULATOR_BASE		0x2000
#define UMP9620_REG_ANA_UMP9622_TSX_CTRL0	0xe080

/* UMP9620 regulator lock register */
#define UMP9620_WR_UNLOCK_VALUE		0x6e7f
#define UMP9620_PWR_WR_PROT		(UMP9620_REGULATOR_BASE + 0x3d0)

/*
 * UMP9620 enable register
 */
#define UMP9620_POWER_PD_SW		(UMP9620_REGULATOR_BASE + 0x01c)
#define UMP9620_DCDC_WPA_PD		(UMP9620_REGULATOR_BASE + 0x0c0)
#define UMP9620_LDO_VDDRF1V8_PD		(UMP9620_REGULATOR_BASE + 0x10c)
#define UMP9620_LDO_VDDCAMIO_PD		(UMP9620_REGULATOR_BASE + 0x118)
#define UMP9620_LDO_VDDWCN_PD		(UMP9620_REGULATOR_BASE + 0x11c)
#define UMP9620_LDO_VDDCAMD1_PD		(UMP9620_REGULATOR_BASE + 0x128)
#define UMP9620_LDO_VDDCAMD0_PD		(UMP9620_REGULATOR_BASE + 0x134)
#define UMP9620_LDO_VDDCAMMOT1_PD	(UMP9620_REGULATOR_BASE + 0x140)
#define UMP9620_LDO_AVDD12_PD		(UMP9620_REGULATOR_BASE + 0x14c)
#define UMP9620_LDO_VDDCAMA0_PD		(UMP9620_REGULATOR_BASE + 0x158)
#define UMP9620_LDO_VDDCAMA1_PD		(UMP9620_REGULATOR_BASE + 0x164)
#define UMP9620_LDO_VDDCAMMOT0_PD	(UMP9620_REGULATOR_BASE + 0x170)
#define UMP9620_LDO_VDDSIM0_PD		(UMP9620_REGULATOR_BASE + 0x17c)
#define UMP9620_LDO_VDDSIM1_PD		(UMP9620_REGULATOR_BASE + 0x188)
#define UMP9620_LDO_VDDSIM2_PD		(UMP9620_REGULATOR_BASE + 0x194)
#define UMP9620_LDO_VDDEMMCCORE_PD	(UMP9620_REGULATOR_BASE + 0x1a0)
#define UMP9620_LDO_VDDSDCORE_PD	(UMP9620_REGULATOR_BASE + 0x1ac)
#define UMP9620_LDO_VDDSDIO_PD		(UMP9620_REGULATOR_BASE + 0x1b8)
#define UMP9620_LDO_VDDWIFIPA_PD	(UMP9620_REGULATOR_BASE + 0x1d0)
#define UMP9620_LDO_VDDUSB33_PD		(UMP9620_REGULATOR_BASE + 0x1e8)
#define UMP9620_LDO_VDDLDO0_PD		(UMP9620_REGULATOR_BASE + 0x1f4)
#define UMP9620_LDO_VDDLDO1_PD		(UMP9620_REGULATOR_BASE + 0x200)
#define UMP9620_LDO_VDDLDO2_PD		(UMP9620_REGULATOR_BASE + 0x20c)
#define UMP9620_LDO_VDDKPLED_PD		(UMP9620_REGULATOR_BASE + 0x38c)
#define UMP9620_LDO_VDDVIB_PD		(UMP9620_REGULATOR_BASE + 0x390)
#define UMP9620_LDO_VDDCAMA2_PD		(UMP9620_REGULATOR_BASE + 0x460)
#define UMP9620_LDO_VDDLDO3_PD		(UMP9620_REGULATOR_BASE + 0x480)
#define UMP9620_LDO_VDDCAMD2_PD		(UMP9620_REGULATOR_BASE + 0x48c)
#define UMP9620_LDO_VDDRF1V1_PD		(UMP9620_REGULATOR_BASE + 0x498)
#define UMP9620_LDO_VDDRF0V9_PD		(UMP9620_REGULATOR_BASE + 0x4a4)
#define UMP9620_LDO_VDDUFS1V2_PD	(UMP9620_REGULATOR_BASE + 0x4b0)
#define UMP9620_LDO_VDDLDO4_PD		(UMP9620_REGULATOR_BASE + 0x4d0)
#define UMP9620_TSX_CTRL2		(UMP9620_REGULATOR_BASE + 0x2f4)
#define UMP9620_LDO_XTL_EN3		(UMP9620_REGULATOR_BASE + 0x2ac)

/*
 * UMP9620 enable mask
 */
#define UMP9620_DCDC_CPU1_PD_MASK		BIT(10)
#define UMP9620_DCDC_CPU2_PD_MASK		BIT(4)
#define UMP9620_DCDC_GPU_PD_MASK		BIT(3)
#define UMP9620_DCDC_CORE_PD_MASK		BIT(5)
#define UMP9620_DCDC_PUB_PD_MASK		BIT(11)
#define UMP9620_DCDC_MEM_PD_MASK		BIT(6)
#define UMP9620_DCDC_MEMQ_PD_MASK		BIT(12)
#define UMP9620_DCDC_GEN0_PD_MASK		BIT(8)
#define UMP9620_DCDC_GEN1_PD_MASK		BIT(7)
#define UMP9620_DCDC_SRAM0_PD_MASK		BIT(13)
#define UMP9620_DCDC_WPA_PD_MASK		BIT(5)
#define UMP9620_LDO_AVDD18_PD_MASK		BIT(2)
#define UMP9620_LDO_VDDRF1V8_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMIO_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDWCN_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMD2_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMD1_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMD0_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDRF0V9_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDRF1V1_PD_MASK		BIT(0)
#define UMP9620_LDO_AVDD12_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMA0_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMA1_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMA2_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMMOT0_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDCAMMOT1_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDSIM0_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDSIM1_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDSIM2_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDEMMCCORE_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDSDCORE_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDSDIO_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDUFS1V2_PD_MASK		BIT(0)
#define UMP9620_LDO_VDD28_PD_MASK		BIT(1)
#define UMP9620_LDO_VDDWIFIPA_PD_MASK		BIT(0)
#define UMP9620_LDO_VDD18_DCXO_PD_MASK		BIT(14)
#define UMP9620_LDO_VDDUSB33_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDLDO0_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDLDO1_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDLDO2_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDLDO3_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDLDO4_PD_MASK		BIT(0)
#define UMP9620_LDO_VDDVIB_PD_MASK		BIT(13)
#define UMP9620_LDO_VDDKPLED_PD_MASK		BIT(15)

/*
 * UMP9620 vsel register
 */
#define UMP9620_DCDC_CPU1_VOL		(UMP9620_REGULATOR_BASE + 0x44c)
#define UMP9620_DCDC_CPU2_VOL		(UMP9620_REGULATOR_BASE + 0x44)
#define UMP9620_DCDC_GPU_VOL		(UMP9620_REGULATOR_BASE + 0x54)
#define UMP9620_DCDC_CORE_VOL		(UMP9620_REGULATOR_BASE + 0x64)
#define UMP9620_DCDC_PUB_VOL		(UMP9620_REGULATOR_BASE + 0x74)
#define UMP9620_DCDC_MEM_VOL		(UMP9620_REGULATOR_BASE + 0x84)
#define UMP9620_DCDC_MEMQ_VOL		(UMP9620_REGULATOR_BASE + 0x94)
#define UMP9620_DCDC_GEN0_VOL		(UMP9620_REGULATOR_BASE + 0xa4)
#define UMP9620_DCDC_GEN1_VOL		(UMP9620_REGULATOR_BASE + 0xb4)
#define UMP9620_DCDC_SRAM0_VOL		(UMP9620_REGULATOR_BASE + 0xdc)
#define UMP9620_DCDC_WPA_VOL		(UMP9620_REGULATOR_BASE + 0xc8)
#define UMP9620_LDO_AVDD18_VOL		(UMP9620_REGULATOR_BASE + 0x104)
#define UMP9620_LDO_VDDRF1V8_VOL	(UMP9620_REGULATOR_BASE + 0x110)
#define UMP9620_LDO_VDDWCN_VOL		(UMP9620_REGULATOR_BASE + 0x120)
#define UMP9620_LDO_VDDCAMD2_VOL	(UMP9620_REGULATOR_BASE + 0x490)
#define UMP9620_LDO_VDDCAMD1_VOL	(UMP9620_REGULATOR_BASE + 0x12c)
#define UMP9620_LDO_VDDCAMD0_VOL	(UMP9620_REGULATOR_BASE + 0x138)
#define UMP9620_LDO_VDDRF0V9_VOL	(UMP9620_REGULATOR_BASE + 0x4a8)
#define UMP9620_LDO_VDDRF1V1_VOL	(UMP9620_REGULATOR_BASE + 0x49c)
#define UMP9620_LDO_AVDD12_VOL		(UMP9620_REGULATOR_BASE + 0x150)
#define UMP9620_LDO_VDDCAMIO_VOL	(UMP9620_REGULATOR_BASE + 0x30)
#define UMP9620_LDO_VDDCAMA0_VOL	(UMP9620_REGULATOR_BASE + 0x15c)
#define UMP9620_LDO_VDDCAMA1_VOL	(UMP9620_REGULATOR_BASE + 0x168)
#define UMP9620_LDO_VDDCAMA2_VOL	(UMP9620_REGULATOR_BASE + 0x464)
#define UMP9620_LDO_VDDCAMMOT0_VOL	(UMP9620_REGULATOR_BASE + 0x174)
#define UMP9620_LDO_VDDCAMMOT1_VOL	(UMP9620_REGULATOR_BASE + 0x144)
#define UMP9620_LDO_VDDSIM0_VOL		(UMP9620_REGULATOR_BASE + 0x180)
#define UMP9620_LDO_VDDSIM1_VOL		(UMP9620_REGULATOR_BASE + 0x18c)
#define UMP9620_LDO_VDDSIM2_VOL		(UMP9620_REGULATOR_BASE + 0x198)
#define UMP9620_LDO_VDDEMMCCORE_VOL	(UMP9620_REGULATOR_BASE + 0x1a4)
#define UMP9620_LDO_VDDSDCORE_VOL	(UMP9620_REGULATOR_BASE + 0x1b0)
#define UMP9620_LDO_VDDSDIO_VOL		(UMP9620_REGULATOR_BASE + 0x1bc)
#define UMP9620_LDO_VDDUFS1V2_VOL	(UMP9620_REGULATOR_BASE + 0x4b4)
#define UMP9620_LDO_VDD28_VOL		(UMP9620_REGULATOR_BASE + 0x1c8)
#define UMP9620_LDO_VDDWIFIPA_VOL	(UMP9620_REGULATOR_BASE + 0x1d4)
#define UMP9620_LDO_VDD18_DCXO_VOL	(UMP9620_REGULATOR_BASE + 0x1e0)
#define UMP9620_LDO_VDDUSB33_VOL	(UMP9620_REGULATOR_BASE + 0x1ec)
#define UMP9620_LDO_VDDLDO0_VOL		(UMP9620_REGULATOR_BASE + 0x1f8)
#define UMP9620_LDO_VDDLDO1_VOL		(UMP9620_REGULATOR_BASE + 0x204)
#define UMP9620_LDO_VDDLDO2_VOL		(UMP9620_REGULATOR_BASE + 0x210)
#define UMP9620_LDO_VDDLDO3_VOL		(UMP9620_REGULATOR_BASE + 0x484)
#define UMP9620_LDO_VDDLDO4_VOL		(UMP9620_REGULATOR_BASE + 0x4d4)
#define UMP9620_LDO_VDDVIB_VOL		(UMP9620_REGULATOR_BASE + 0x390)
#define UMP9620_LDO_VDDKPLED_VOL	(UMP9620_REGULATOR_BASE + 0x38c)

/*
 * UMP9620 vsel register mask
 */
#define UMP9620_DCDC_CPU1_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_CPU2_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_GPU_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_CORE_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_PUB_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_MEM_VOL_MASK		GENMASK(7, 0)
#define UMP9620_DCDC_MEMQ_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_GEN0_VOL_MASK		GENMASK(7, 0)
#define UMP9620_DCDC_GEN1_VOL_MASK		GENMASK(7, 0)
#define UMP9620_DCDC_SRAM0_VOL_MASK		GENMASK(8, 0)
#define UMP9620_DCDC_WPA_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_AVDD18_VOL_MASK		GENMASK(5, 0)
#define UMP9620_LDO_VDDRF1V8_VOL_MASK		GENMASK(5, 0)
#define UMP9620_LDO_VDDWCN_VOL_MASK		GENMASK(5, 0)
#define UMP9620_LDO_VDDCAMD2_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDDCAMD1_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDDCAMD0_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDDRF0V9_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDDRF1V1_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_AVDD12_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDDCAMIO_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDCAMA0_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDCAMA1_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDCAMA2_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDCAMMOT0_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDCAMMOT1_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDSIM0_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDSIM1_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDSIM2_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDEMMCCORE_VOL_MASK	GENMASK(7, 0)
#define UMP9620_LDO_VDDSDCORE_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDSDIO_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDUFS1V2_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDD28_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDWIFIPA_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDD18_DCXO_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDUSB33_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDLDO0_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDLDO1_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDLDO2_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDLDO3_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDLDO4_VOL_MASK		GENMASK(4, 0)
#define UMP9620_LDO_VDDVIB_VOL_MASK		GENMASK(7, 0)
#define UMP9620_LDO_VDDKPLED_VOL_MASK		GENMASK(14, 7)

#define SPRD_MIN_VOLTAGE_UV	300000

enum ump9620_regulator_id {
	UMP9620_DCDC_CPU1,
	UMP9620_DCDC_CPU2,
	UMP9620_DCDC_GPU,
	UMP9620_DCDC_CORE,
	UMP9620_DCDC_PUB,
	UMP9620_DCDC_MEM,
	UMP9620_DCDC_MEMQ,
	UMP9620_DCDC_GEN0,
	UMP9620_DCDC_GEN1,
	UMP9620_DCDC_SRAM0,
	UMP9620_DCDC_WPA,
	UMP9620_LDO_AVDD18,
	UMP9620_LDO_VDDRF1V8,
	UMP9620_LDO_VDDCAMIO,
	UMP9620_LDO_VDDWCN,
	UMP9620_LDO_VDDCAMD2,
	UMP9620_LDO_VDDCAMD1,
	UMP9620_LDO_VDDCAMD0,
	UMP9620_LDO_VDDRF0V9,
	UMP9620_LDO_VDDRF1V1,
	UMP9620_LDO_AVDD12,
	UMP9620_LDO_VDDCAMA0,
	UMP9620_LDO_VDDCAMA1,
	UMP9620_LDO_VDDCAMA2,
	UMP9620_LDO_VDDCAMMOT0,
	UMP9620_LDO_VDDCAMMOT1,
	UMP9620_LDO_VDDSIM0,
	UMP9620_LDO_VDDSIM1,
	UMP9620_LDO_VDDSIM2,
	UMP9620_LDO_VDDEMMCCORE,
	UMP9620_LDO_VDDSDCORE,
	UMP9620_LDO_VDDSDIO,
	UMP9620_LDO_VDDUFS1V2,
	UMP9620_LDO_VDD28,
	UMP9620_LDO_VDDWIFIPA,
	UMP9620_LDO_VDD18_DCXO,
	UMP9620_LDO_VDDUSB33,
	UMP9620_LDO_VDDLDO0,
	UMP9620_LDO_VDDLDO1,
	UMP9620_LDO_VDDLDO2,
	UMP9620_LDO_VDDLDO3,
	UMP9620_LDO_VDDLDO4,
	UMP9620_LDO_VDDVIB,
	UMP9620_LDO_VDDKPLED,
	UMP9620_CLK_REFOUT1,
	UMP9620_CLK_REFOUT2_XTL2,
	UMP9620_LDO_VDDSIM2_XTL2,
};

static struct dentry *debugfs_root;

static int regulator_set_voltage_sel_sprd(struct regulator_dev *rdev, unsigned int sel)
{
	if (!rdev->desc->min_uV) {
		if (sel < (SPRD_MIN_VOLTAGE_UV / rdev->desc->uV_step)) {
			char kevent[] = "kevent_begin:{\"event_id\":\"107000001\",\"event_time\"";
			char *pr_str[2];

			pr_str[0] = kasprintf(GFP_KERNEL,
					      "%s:%ld,\"task\":%s,\"PID\":%d,\"%s[0x%04x]\":0x%04x}:kevent_end\n",
					      kevent, jiffies, current->comm, current->pid,
					      rdev->desc->name, rdev->desc->vsel_reg, sel);
			pr_str[1] = NULL;
			kobject_uevent_env(&rdev->dev.kobj, KOBJ_CHANGE, pr_str);
			kfree(pr_str[0]);
		}
	}
	return regulator_set_voltage_sel_regmap(rdev, sel);
};

/* only for ump962x Project */
static int sprd_regulator_disable_regmap(struct regulator_dev *rdev)
{
	int id = 0, ret = 0;

	id = rdev_get_id(rdev);
	if (id == UMP9620_LDO_VDDSDIO || id == UMP9620_LDO_VDDSDCORE) {
		ret = regulator_disable_regmap(rdev);
		usleep_range(10 * 1000, 10 * 1250);
		return ret;
	} else {
		return regulator_disable_regmap(rdev);
	}
};

static const struct regulator_ops ump9620_regu_linear_ops = {
	.enable = regulator_enable_regmap,
	.disable = sprd_regulator_disable_regmap,
	.is_enabled = regulator_is_enabled_regmap,
	.list_voltage = regulator_list_voltage_linear,
	.get_voltage_sel = regulator_get_voltage_sel_regmap,
	.set_voltage_sel = regulator_set_voltage_sel_sprd,
};

static const struct regulator_ops ump9620_nopower_linear_ops = {
	.enable = regulator_enable_regmap,
	.disable = regulator_disable_regmap,
	.is_enabled = regulator_is_enabled_regmap,
};

#define UMP9620_REGU_LINEAR(_id, en_reg, en_mask, vreg, vmask,	\
			  vstep, vmin, vmax, min_sel) {		\
	.name			= #_id,				\
	.of_match		= of_match_ptr(#_id),		\
	.ops			= &ump9620_regu_linear_ops,	\
	.type			= REGULATOR_VOLTAGE,		\
	.id			= UMP9620_##_id,		\
	.owner			= THIS_MODULE,			\
	.min_uV			= vmin,				\
	.n_voltages		= ((vmax) - (vmin)) / (vstep) + 1,	\
	.uV_step		= vstep,			\
	.enable_is_inverted	= true,				\
	.enable_val		= 0,				\
	.enable_reg		= en_reg,			\
	.enable_mask		= en_mask,			\
	.vsel_reg		= vreg,				\
	.vsel_mask		= vmask,			\
	.linear_min_sel		= min_sel,			\
}

#define UMP9620_NOPOWER_LINEAR(_id, en_reg, en_mask, vreg, vmask,		\
			  vstep, vmin, vmax, min_sel, en_val) {			\
	.name			= #_id,						\
	.of_match		= of_match_ptr(#_id),				\
	.ops			= &ump9620_nopower_linear_ops,			\
	.type			= REGULATOR_VOLTAGE,				\
	.fixed_uV		= 1800000,					\
	.id			= UMP9620_##_id,				\
	.owner			= THIS_MODULE,					\
	.min_uV			= vmin,						\
	.n_voltages		= 1,						\
	.uV_step		= vstep,					\
	.enable_is_inverted	= false,					\
	.enable_val		= en_val,					\
	.enable_reg		= en_reg,					\
	.enable_mask		= en_mask,					\
	.vsel_reg		= vreg,						\
	.vsel_mask		= vmask,					\
	.linear_min_sel		= min_sel,					\
}

static struct regulator_desc regulators[] = {
	UMP9620_REGU_LINEAR(DCDC_CPU1, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_CPU1_PD_MASK, UMP9620_DCDC_CPU1_VOL,
			    UMP9620_DCDC_CPU1_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_CPU2, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_CPU2_PD_MASK, UMP9620_DCDC_CPU2_VOL,
			    UMP9620_DCDC_CPU2_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_GPU, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_GPU_PD_MASK, UMP9620_DCDC_GPU_VOL,
			    UMP9620_DCDC_GPU_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_CORE, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_CORE_PD_MASK, UMP9620_DCDC_CORE_VOL,
			    UMP9620_DCDC_CORE_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_PUB, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_PUB_PD_MASK, UMP9620_DCDC_PUB_VOL,
			    UMP9620_DCDC_PUB_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_MEM, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_MEM_PD_MASK, UMP9620_DCDC_MEM_VOL,
			    UMP9620_DCDC_MEM_VOL_MASK, 6250, 0, 1593750, 0),
	UMP9620_REGU_LINEAR(DCDC_MEMQ, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_MEMQ_PD_MASK, UMP9620_DCDC_MEMQ_VOL,
			    UMP9620_DCDC_MEMQ_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_GEN0, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_GEN0_PD_MASK, UMP9620_DCDC_GEN0_VOL,
			    UMP9620_DCDC_GEN0_VOL_MASK, 9375, 0, 2390625, 0),
	UMP9620_REGU_LINEAR(DCDC_GEN1, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_GEN1_PD_MASK, UMP9620_DCDC_GEN1_VOL,
			    UMP9620_DCDC_GEN1_VOL_MASK, 6250, 0, 1593750, 0),
	UMP9620_REGU_LINEAR(DCDC_SRAM0, UMP9620_POWER_PD_SW,
			    UMP9620_DCDC_SRAM0_PD_MASK, UMP9620_DCDC_SRAM0_VOL,
			    UMP9620_DCDC_SRAM0_VOL_MASK, 3125, 0, 1596875, 0),
	UMP9620_REGU_LINEAR(DCDC_WPA, UMP9620_DCDC_WPA_PD,
			    UMP9620_DCDC_WPA_PD_MASK, UMP9620_DCDC_WPA_VOL,
			    UMP9620_DCDC_WPA_VOL_MASK, 14100, 0, 3595500, 0),
	UMP9620_REGU_LINEAR(LDO_AVDD18, UMP9620_POWER_PD_SW,
			    UMP9620_LDO_AVDD18_PD_MASK, UMP9620_LDO_AVDD18_VOL,
			    UMP9620_LDO_AVDD18_VOL_MASK, 10000, 1175000, 1805000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDRF1V8, UMP9620_LDO_VDDRF1V8_PD,
			    UMP9620_LDO_VDDRF1V8_PD_MASK, UMP9620_LDO_VDDRF1V8_VOL,
			    UMP9620_LDO_VDDRF1V8_VOL_MASK, 10000, 1175000, 1805000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDWCN, UMP9620_LDO_VDDWCN_PD,
			    UMP9620_LDO_VDDWCN_PD_MASK, UMP9620_LDO_VDDWCN_VOL,
			    UMP9620_LDO_VDDWCN_VOL_MASK, 15000, 900000, 1845000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMD2, UMP9620_LDO_VDDCAMD2_PD,
			    UMP9620_LDO_VDDCAMD2_PD_MASK, UMP9620_LDO_VDDCAMD2_VOL,
			    UMP9620_LDO_VDDCAMD2_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMD1, UMP9620_LDO_VDDCAMD1_PD,
			    UMP9620_LDO_VDDCAMD1_PD_MASK, UMP9620_LDO_VDDCAMD1_VOL,
			    UMP9620_LDO_VDDCAMD1_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMD0, UMP9620_LDO_VDDCAMD0_PD,
			    UMP9620_LDO_VDDCAMD0_PD_MASK, UMP9620_LDO_VDDCAMD0_VOL,
			    UMP9620_LDO_VDDCAMD0_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDRF0V9, UMP9620_LDO_VDDRF0V9_PD,
			    UMP9620_LDO_VDDRF0V9_PD_MASK, UMP9620_LDO_VDDRF0V9_VOL,
			    UMP9620_LDO_VDDRF0V9_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDRF1V1, UMP9620_LDO_VDDRF1V1_PD,
			    UMP9620_LDO_VDDRF1V1_PD_MASK, UMP9620_LDO_VDDRF1V1_VOL,
			    UMP9620_LDO_VDDRF1V1_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_AVDD12, UMP9620_LDO_AVDD12_PD,
			    UMP9620_LDO_AVDD12_PD_MASK, UMP9620_LDO_AVDD12_VOL,
			    UMP9620_LDO_AVDD12_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMA0, UMP9620_LDO_VDDCAMA0_PD,
			    UMP9620_LDO_VDDCAMA0_PD_MASK, UMP9620_LDO_VDDCAMA0_VOL,
			    UMP9620_LDO_VDDCAMA0_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMA1, UMP9620_LDO_VDDCAMA1_PD,
			    UMP9620_LDO_VDDCAMA1_PD_MASK, UMP9620_LDO_VDDCAMA1_VOL,
			    UMP9620_LDO_VDDCAMA1_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMA2, UMP9620_LDO_VDDCAMA2_PD,
			    UMP9620_LDO_VDDCAMA2_PD_MASK, UMP9620_LDO_VDDCAMA2_VOL,
			    UMP9620_LDO_VDDCAMA2_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMMOT0, UMP9620_LDO_VDDCAMMOT0_PD,
			    UMP9620_LDO_VDDCAMMOT0_PD_MASK, UMP9620_LDO_VDDCAMMOT0_VOL,
			    UMP9620_LDO_VDDCAMMOT0_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMMOT1, UMP9620_LDO_VDDCAMMOT1_PD,
			    UMP9620_LDO_VDDCAMMOT1_PD_MASK, UMP9620_LDO_VDDCAMMOT1_VOL,
			    UMP9620_LDO_VDDCAMMOT1_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDSIM0, UMP9620_LDO_VDDSIM0_PD,
			    UMP9620_LDO_VDDSIM0_PD_MASK, UMP9620_LDO_VDDSIM0_VOL,
			    UMP9620_LDO_VDDSIM0_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDSIM1, UMP9620_LDO_VDDSIM1_PD,
			    UMP9620_LDO_VDDSIM1_PD_MASK, UMP9620_LDO_VDDSIM1_VOL,
			    UMP9620_LDO_VDDSIM1_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDSIM2, UMP9620_LDO_VDDSIM2_PD,
			    UMP9620_LDO_VDDSIM2_PD_MASK, UMP9620_LDO_VDDSIM2_VOL,
			    UMP9620_LDO_VDDSIM2_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDEMMCCORE, UMP9620_LDO_VDDEMMCCORE_PD,
			    UMP9620_LDO_VDDEMMCCORE_PD_MASK, UMP9620_LDO_VDDEMMCCORE_VOL,
			    UMP9620_LDO_VDDEMMCCORE_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDSDCORE, UMP9620_LDO_VDDSDCORE_PD,
			    UMP9620_LDO_VDDSDCORE_PD_MASK, UMP9620_LDO_VDDSDCORE_VOL,
			    UMP9620_LDO_VDDSDCORE_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDSDIO, UMP9620_LDO_VDDSDIO_PD,
			    UMP9620_LDO_VDDSDIO_PD_MASK, UMP9620_LDO_VDDSDIO_VOL,
			    UMP9620_LDO_VDDSDIO_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDUFS1V2, UMP9620_LDO_VDDUFS1V2_PD,
			    UMP9620_LDO_VDDUFS1V2_PD_MASK, UMP9620_LDO_VDDUFS1V2_VOL,
			    UMP9620_LDO_VDDUFS1V2_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDD28, UMP9620_POWER_PD_SW,
			    UMP9620_LDO_VDD28_PD_MASK, UMP9620_LDO_VDD28_VOL,
			    UMP9620_LDO_VDD28_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDWIFIPA, UMP9620_LDO_VDDWIFIPA_PD,
			    UMP9620_LDO_VDDWIFIPA_PD_MASK, UMP9620_LDO_VDDWIFIPA_VOL,
			    UMP9620_LDO_VDDWIFIPA_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDD18_DCXO, UMP9620_POWER_PD_SW,
			    UMP9620_LDO_VDD18_DCXO_PD_MASK, UMP9620_LDO_VDD18_DCXO_VOL,
			    UMP9620_LDO_VDD18_DCXO_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDUSB33, UMP9620_LDO_VDDUSB33_PD,
			    UMP9620_LDO_VDDUSB33_PD_MASK, UMP9620_LDO_VDDUSB33_VOL,
			    UMP9620_LDO_VDDUSB33_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDLDO0, UMP9620_LDO_VDDLDO0_PD,
			    UMP9620_LDO_VDDLDO0_PD_MASK, UMP9620_LDO_VDDLDO0_VOL,
			    UMP9620_LDO_VDDLDO0_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDLDO1, UMP9620_LDO_VDDLDO1_PD,
			    UMP9620_LDO_VDDLDO1_PD_MASK, UMP9620_LDO_VDDLDO1_VOL,
			    UMP9620_LDO_VDDLDO1_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDLDO2, UMP9620_LDO_VDDLDO2_PD,
			    UMP9620_LDO_VDDLDO2_PD_MASK, UMP9620_LDO_VDDLDO2_VOL,
			    UMP9620_LDO_VDDLDO2_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDLDO3, UMP9620_LDO_VDDLDO3_PD,
			    UMP9620_LDO_VDDLDO3_PD_MASK, UMP9620_LDO_VDDLDO3_VOL,
			    UMP9620_LDO_VDDLDO3_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDLDO4, UMP9620_LDO_VDDLDO4_PD,
			    UMP9620_LDO_VDDLDO4_PD_MASK, UMP9620_LDO_VDDLDO4_VOL,
			    UMP9620_LDO_VDDLDO4_VOL_MASK, 15000, 900000, 1365000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDVIB, UMP9620_LDO_VDDVIB_PD,
			    UMP9620_LDO_VDDVIB_PD_MASK, UMP9620_LDO_VDDVIB_VOL,
			    UMP9620_LDO_VDDVIB_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDKPLED, UMP9620_LDO_VDDKPLED_PD,
			    UMP9620_LDO_VDDKPLED_PD_MASK, UMP9620_LDO_VDDKPLED_VOL,
			    UMP9620_LDO_VDDKPLED_VOL_MASK, 10000, 1200000, 3750000, 0),
	UMP9620_REGU_LINEAR(LDO_VDDCAMIO, UMP9620_LDO_VDDCAMIO_PD,
			    UMP9620_LDO_VDDCAMIO_PD_MASK, UMP9620_LDO_VDDCAMIO_VOL,
			    UMP9620_LDO_VDDCAMIO_VOL_MASK, 10000, 1200000, 1830000, 0),
	UMP9620_NOPOWER_LINEAR(CLK_REFOUT1, UMP9620_REG_ANA_UMP9622_TSX_CTRL0,
			    0x2, UMP9620_LDO_VDDLDO3_VOL, UMP9620_LDO_VDDLDO3_VOL_MASK,
			    10000, 1200000, 3750000, 0, 0x2),
	UMP9620_NOPOWER_LINEAR(CLK_REFOUT2_XTL2, UMP9620_TSX_CTRL2,
			    0x4, UMP9620_LDO_VDDLDO3_VOL, UMP9620_LDO_VDDLDO3_VOL_MASK,
			    10000, 1200000, 3750000, 0, 0x4),
	UMP9620_NOPOWER_LINEAR(LDO_VDDSIM2_XTL2, UMP9620_LDO_XTL_EN3,
			    0x400, UMP9620_LDO_VDDLDO3_VOL, UMP9620_LDO_VDDLDO3_VOL_MASK,
			    10000, 1200000, 3750000, 0, 0x400),
};

static int debugfs_enable_get(void *data, u64 *val)
{
	struct regulator_dev *rdev = data;

	if (rdev && rdev->desc->ops->is_enabled)
		*val = rdev->desc->ops->is_enabled(rdev);
	else
		*val = ~0ULL;
	return 0;
}

static int debugfs_enable_set(void *data, u64 val)
{
	struct regulator_dev *rdev = data;

	if (rdev && rdev->desc->ops->enable && rdev->desc->ops->disable) {
		if (val)
			rdev->desc->ops->enable(rdev);
		else
			rdev->desc->ops->disable(rdev);
	}

	return 0;
}

static int debugfs_voltage_get(void *data, u64 *val)
{
	int sel, ret;
	struct regulator_dev *rdev = data;

	if (!rdev || !rdev->desc->ops->get_voltage_sel || !rdev->desc->ops->list_voltage)
		return -EINVAL;

	sel = rdev->desc->ops->get_voltage_sel(rdev);
	if (sel < 0)
		return sel;

	ret = rdev->desc->ops->list_voltage(rdev, sel);
	*val = ret / 1000;

	return 0;
}

static int debugfs_voltage_set(void *data, u64 val)
{
	int selector;
	struct regulator_dev *rdev = data;

	if (!rdev || !rdev->desc->ops->set_voltage_sel)
		return -EINVAL;

	val = val * 1000;
	if (val < rdev->desc->min_uV)
		return -EINVAL;
	selector = regulator_map_voltage_linear(rdev,
						val - rdev->desc->uV_step / 2,
						val + rdev->desc->uV_step / 2);

	if (selector < 0)
		return selector;

	return rdev->desc->ops->set_voltage_sel(rdev, selector);
}

DEFINE_SIMPLE_ATTRIBUTE(fops_enable, debugfs_enable_get, debugfs_enable_set, "%llu\n");
DEFINE_SIMPLE_ATTRIBUTE(fops_ldo, debugfs_voltage_get, debugfs_voltage_set, "%llu\n");

static void ump9620_regulator_debugfs_init(struct regulator_dev *rdev)
{
	struct dentry *debugfs;

	if (IS_ERR_OR_NULL(debugfs_root)) {
		debugfs_root = debugfs_create_dir("sprd_regulator", NULL);
		if (IS_ERR_OR_NULL(debugfs_root)) {
			dev_warn(&rdev->dev, "Failed to recreate debugfs directory\n");
			return;
		}
	}

	debugfs = debugfs_create_dir(rdev->desc->name, debugfs_root);
	if (IS_ERR_OR_NULL(debugfs)) {
		dev_warn(&rdev->dev, "Failed to create %s directory\n", rdev->desc->name);
		return;
	}

	debugfs_create_file("enable", 0644, debugfs, rdev, &fops_enable);
	debugfs_create_file("voltage", 0644, debugfs, rdev, &fops_ldo);
}

static int ump9620_regulator_unlock(struct regmap *regmap)
{
	return regmap_write(regmap, UMP9620_PWR_WR_PROT, UMP9620_WR_UNLOCK_VALUE);
}

static int ump9620_regulator_probe(struct platform_device *pdev)
{
	int i, ret;
	struct regmap *regmap;
	struct regulator_config config = { };
	struct regulator_dev *rdev;

	regmap = dev_get_regmap(pdev->dev.parent, NULL);
	if (!regmap) {
		dev_err(&pdev->dev, "failed to get regmap.\n");
		return -ENODEV;
	}

	ret = ump9620_regulator_unlock(regmap);
	if (ret) {
		dev_err(&pdev->dev, "failed to release regulator lock\n");
		return ret;
	}

	config.dev = &pdev->dev;
	config.regmap = regmap;

	debugfs_root = debugfs_lookup("sprd_regulator", NULL);
	if (!debugfs_root)
		debugfs_root = debugfs_create_dir("sprd_regulator", NULL);

	if (IS_ERR(debugfs_root))
		dev_warn(&pdev->dev, "Failed to create debugfs directory\n");

	for (i = 0; i < ARRAY_SIZE(regulators); i++) {
		rdev = devm_regulator_register(&pdev->dev, &regulators[i], &config);
		if (IS_ERR(rdev)) {
			dev_err(&pdev->dev, "failed to register regulator %s\n",
				regulators[i].name);
			continue;
		}
		ump9620_regulator_debugfs_init(rdev);
	}

	return 0;
}

static int ump9620_regulator_remove(struct platform_device *pdev)
{
	debugfs_remove_recursive(debugfs_root);
	return 0;
}

static const struct of_device_id ump9620_regulator_match[] = {
	{ .compatible = "sprd,ump9620-regulator" },
	{},
};
MODULE_DEVICE_TABLE(of, ump9620_regulator_match);

static struct platform_driver ump9620_regulator_driver = {
	.driver = {
		   .name = "ump962x-regulator",
		   .of_match_table = ump9620_regulator_match,
		   },
	.probe = ump9620_regulator_probe,
	.remove = ump9620_regulator_remove,
};

module_platform_driver(ump9620_regulator_driver);

MODULE_AUTHOR("Zhongfa.Wang <zhongfa.wang@unisoc.com>");
MODULE_DESCRIPTION("Unisoc ump9620 regulator driver");
MODULE_LICENSE("GPL v2");
