#!/usr/bin/awk -f
# SPDX-License-Identifier: ISC
#
# genregdb.awk -- generate regdb.c from db.txt
#
# Actually, it reads from stdin (presumed to be db.txt) and writes
# to stdout (presumed to be regdb.c), but close enough...
#
# Copyright 2009 John W. Linville <linville@tuxdriver.com>

BEGIN {
	active = 0
	rules = 0;
	print "/*"
	print " * DO NOT EDIT -- file generated from data in db.txt"
	print " */"
	print ""
	print "#include <linux/nl80211.h>"
	print "#include <net/cfg80211.h>"
	print "#include \"regdb.h\""
	print ""
	regdb = "const struct ieee80211_regdomain *reg_regdb[] = {\n"
}

function parse_country_head() {
	country=$2
	sub(/:/, "", country)
	printf "static const struct ieee80211_regdomain regdom_%s = {\n", country
	printf "\t.alpha2 = \"%s\",\n", country
	if ($NF ~ /DFS-ETSI/)
		printf "\t.dfs_region = NL80211_DFS_ETSI,\n"
	else if ($NF ~ /DFS-FCC/)
		printf "\t.dfs_region = NL80211_DFS_FCC,\n"
	else if ($NF ~ /DFS-JP/)
		printf "\t.dfs_region = NL80211_DFS_JP,\n"
	printf "\t.reg_rules = {\n"
	active = 1
	regdb = regdb "\t&regdom_" country ",\n"
}

function parse_reg_rule()
{
	flag_starts_at = 7

	start = $1
	sub(/\(/, "", start)
	end = $3
	bw = $5
	sub(/\),/, "", bw)
	gain = 0
	power = $6
	# power might be in mW...
	units = $7
	dfs_cac = 0

	sub(/\(/, "", power)
	sub(/\),/, "", power)
	sub(/\),/, "", units)
	sub(/\)/, "", units)

	if (units == "mW") {
		flag_starts_at = 8
		power = 10 * log(power)/log(10)
		if ($8 ~ /[[:digit:]]/) {
			flag_starts_at = 9
			dfs_cac = $8
		}
	} else {
		if ($7 ~ /[[:digit:]]/) {
			flag_starts_at = 8
			dfs_cac = $7
		}
	}
	sub(/\(/, "", dfs_cac)
	sub(/\),/, "", dfs_cac)
	flagstr = ""
	for (i=flag_starts_at; i<=NF; i++)
		flagstr = flagstr $i
	split(flagstr, flagarray, ",")
	flags = ""
	for (arg in flagarray) {
		if (flagarray[arg] == "NO-OFDM") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_OFDM | "
		} else if (flagarray[arg] == "NO-CCK") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_CCK | "
		} else if (flagarray[arg] == "NO-INDOOR") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_INDOOR | "
		} else if (flagarray[arg] == "NO-OUTDOOR") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_OUTDOOR | "
		} else if (flagarray[arg] == "DFS") {
			flags = flags "\n\t\t\tNL80211_RRF_DFS | "
		} else if (flagarray[arg] == "PTP-ONLY") {
			flags = flags "\n\t\t\tNL80211_RRF_PTP_ONLY | "
		} else if (flagarray[arg] == "PTMP-ONLY") {
			flags = flags "\n\t\t\tNL80211_RRF_PTMP_ONLY | "
		} else if (flagarray[arg] == "PASSIVE-SCAN") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_IR | "
		} else if (flagarray[arg] == "NO-IBSS") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_IR | "
		} else if (flagarray[arg] == "NO-IR") {
			flags = flags "\n\t\t\tNL80211_RRF_NO_IR | "
		} else if (flagarray[arg] == "AUTO-BW") {
			flags = flags "\n\t\t\tNL80211_RRF_AUTO_BW | "
		}

	}
	flags = flags "0"
	printf "\t\tREG_RULE_EXT(%d, %d, %d, %d, %.0f, %d, %s),\n", start, end, bw, gain, power, dfs_cac, flags
	rules++
}

function print_tail_country()
{
	active = 0
	printf "\t},\n"
	printf "\t.n_reg_rules = %d\n", rules
	printf "};\n\n"
	rules = 0;
}

/^[ \t]*#/ {
	# Ignore
}

!active && /^[ \t]*$/ {
	# Ignore
}

!active && /country/ {
	parse_country_head()
}

active && /^[ \t]*\(/ {
	parse_reg_rule()
}

active && /^[ \t]*$/ {
	print_tail_country()
}

END {
	if (active)
		print_tail_country()
	print regdb "};"
	print ""
	print "int reg_regdb_size = ARRAY_SIZE(reg_regdb);"
}
