/* sgm7220.c -- SGMICRO SGM7220 USB TYPE-C Controller device driver */
/* Copyright (c) 2009-2019, Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#define pr_fmt(fmt)	"[cne-charge][sgm7220] %s: " fmt, __func__

#define __SGM7220_ADD_PR_SWITCH__
#define __SGM7220_ADD_ADAPTER__
#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>
#include <linux/workqueue.h>
#include <linux/pm_runtime.h>
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/power_supply.h>
#include <linux/proc_fs.h>
#include <linux/usb/typec.h>
#include <linux/extcon-provider.h>
#include <linux/pinctrl/consumer.h>
#include <uapi/linux/sched/types.h>
#if defined(__SGM7220_ADD_ADAPTER__)
#include "adapter_class.h"
#include "../class.h"

#endif

#if defined(__SGM7220_ADD_PR_SWITCH__)
#define SGM7220_ROLE_REVERSAL_DELAY_MS		2000
#endif

/******************************************************************************
 * 
 * Register addresses
 * 
 *****************************************************************************/
#define REG_BASE                    0x00
/* 0x00 - 0x07 reserved for ID */
#define REG_MOD                     0x08
#define REG_INT                     0x09
#define REG_SET                     0x0A
#define REG_CTL                     0X45
#define REG_REVISION                0XA0

/******************************************************************************
 * 
 * Register bits
 *
 ******************************************************************************/
/* REG_ID (0x00) */
#define ID_REG_LEN  0X08
static char SGM7220_ID[ID_REG_LEN + 1] = "TUSB320"; 
/* REG_MOD (0x08) */
#define MOD_ACTIVE_CABLE_DETECTION  0X01    /*RU*/
#define MOD_ACCESSORY_CONNECTED_SHIFT   1
#define MOD_ACCESSORY_CONNECTED         (0x07 << MOD_ACCESSORY_CONNECTED_SHIFT)    /*RU*/
#define MOD_CURRENT_MODE_DETECT_SHIFT    4
#define MOD_CURRENT_MODE_DETECT         (0x03 << MOD_CURRENT_MODE_DETECT_SHIFT)    /*RU*/
#define MOD_CURRENT_MODE_ADVERTISE_SHIFT    6
#define MOD_CURRENT_MODE_ADVERTISE          (0x03 << MOD_CURRENT_MODE_ADVERTISE_SHIFT)    /*RW*/
/* REG_INT (0x09) */
#define INT_DRP_DUTY_CYCLE_SHIFT    1
#define INT_DRP_DUTY_CYCLE          (0x03 << INT_DRP_DUTY_CYCLE_SHIFT)      /*RW*/
#define INT_INTERRUPT_STATUS_SHIFT  4
#define INT_INTERRUPT_STATUS        (0x01 << INT_INTERRUPT_STATUS_SHIFT)    /*RCU*/
#define INT_CABLE_DIR_SHIFT         5
#define INT_CABLE_DIR               (0x01 << INT_CABLE_DIR_SHIFT)           /*RU*/
#define INT_ATTACHED_STATE_SHIFT    6
#define INT_ATTACHED_STATE          (0x03 << INT_ATTACHED_STATE_SHIFT)      /*RU*/

/* REG_SET (0x0A) */
#define SET_I2C_DISABLE_TERM        0x01    /*RW*/
#define SET_I2C_SOURCE_PREF_SHIFT   1
#define SET_I2C_SOURCE_PREF         (0x03 << SET_I2C_SOURCE_PREF_SHIFT)  /*RW*/
#define SET_I2C_SOFT_RESET_SHIFT    3
#define SET_I2C_SOFT_RESET          (0x01 << SET_I2C_SOFT_RESET_SHIFT)  /*RSU*/
#define SET_MODE_SELECT_SHIFT       4
#define SET_MODE_SELECT             (0x03 << SET_MODE_SELECT_SHIFT)     /*RW*/
#define SET_DEBOUNCE_SHIFT          6
#define SET_DEBOUNCE                (0x03 << SET_DEBOUNCE_SHIFT)        /*RW*/

/* REG_CTR (0x45) */
#define CTR_DISABLE_RD_RP_SHIFT     2
#define CTR_DISABLE_RD_RP           (0x01 << CTR_DISABLE_RD_RP_SHIFT)   /*RW*/

/******************************************************************************
 *
 * Register values
 *
 ******************************************************************************/
/* SET_MODE_SELECT */
#define SET_MODE_SELECT_DEFAULT  0x00
#define SET_MODE_SELECT_SNK       0x01
#define SET_MODE_SELECT_SRC       0x02
#define SET_MODE_SELECT_DRP       0x03
/* MOD_CURRENT_MODE_ADVERTISE */
#define MOD_CURRENT_MODE_ADVERTISE_DEFAULT      0x00
#define MOD_CURRENT_MODE_ADVERTISE_MID          0x01
#define MOD_CURRENT_MODE_ADVERTISE_HIGH         0x02
/* MOD_CURRENT_MODE_DETECT */
#define MOD_CURRENT_MODE_DETECT_DEFAULT      0x00
#define MOD_CURRENT_MODE_DETECT_MID          0x01
#define MOD_CURRENT_MODE_DETECT_ACCESSARY    0x02
#define MOD_CURRENT_MODE_DETECT_HIGH         0x03

#define SGM7220_NAME "SGM7220"
char typec_info[32]="unknown";
EXPORT_SYMBOL(typec_info);

/******************************************************************************
 *
 * Constants
 *
 ******************************************************************************/
enum drp_toggle_type {
	TOGGLE_DFP_DRP_30 = 0,
	TOGGLE_DFP_DRP_40,
	TOGGLE_DFP_DRP_50,
	TOGGLE_DFP_DRP_60
};

enum current_adv_type {
	HOST_CUR_USB = 0,   /*default 500mA or 900mA*/
	HOST_CUR_1P5,      /*1.5A*/
	HOST_CUR_3A       /*3A*/
};

enum current_det_type {
	DET_CUR_USB = 0,    /*default 500mA or 900mA*/
	DET_CUR_1P5,
	DET_CUR_ACCESSORY,  /*charg through accessory 500mA*/
	DET_CUR_3A
};

enum accessory_attach_type {
	ACCESSORY_NOT_ATTACHED = 0,
	ACCESSORY_AUDIO = 4,
	ACCESSORY_CHG_THRU_AUDIO = 5,
	ACCESSORY_DEBUG = 6
};

enum cable_attach_type {
	CABLE_NOT_ATTACHED = 0,
	CABLE_ATTACHED
};

enum cable_state_type {
	CABLE_STATE_NOT_ATTACHED = 0,
	CABLE_STATE_AS_DFP,
	CABLE_STATE_AS_UFP,
	CABLE_STATE_TO_ACCESSORY
};

enum cable_dir_type {
	ORIENT_CC2 = 0,
	ORIENT_CC1
};

enum cc_modes_type {
	MODE_DEFAULT = 0,
	MODE_UFP,
	MODE_DFP,
	MODE_DRP
};

/* Type-C Attrs */
struct type_c_parameters {
	enum current_det_type current_det;         /*charging current on UFP*/
	enum accessory_attach_type accessory_attach;     /*if an accessory is attached*/
	enum cable_attach_type active_cable_attach;         /*if an active_cable is attached*/
	enum cable_state_type attach_state;        /*DFP->UFP or UFP->DFP*/
	enum cable_dir_type cable_dir;           /*cc1 or cc2*/
};

struct state_disorder_monitor {
	int count;
	int err_detected;
	enum cable_state_type former_state;
	unsigned long time_before;
};

enum en_gpio_type {
	EN_CC_MUX = 0,
	EN_VREG_5V,
	EN_GPIO_MAX
};

struct en_gpio_item {
	char *name;
	int index;
};

static struct en_gpio_item en_gpio_info[] = {
	{
		.name   = "sgm7220,en_cc_gpio",
		.index  = EN_CC_MUX,
	},
	{
		.name   = "sgm7220,vreg_5v_gpio",
		.index  = EN_VREG_5V,
	},
	/*add more gpio here if need*/
};

struct sgm7220_info {
	struct i2c_client  *i2c;
	struct device *dev;
	struct mutex  mutex;

	int irq_gpio;
	int en_gpio[EN_GPIO_MAX];

	struct type_c_parameters type_c_param;
	struct state_disorder_monitor monitor;

	struct typec_port *port;
	struct typec_capability typec_cap;
	struct typec_partner *partner;

	struct kthread_worker irq_worker;
	struct kthread_delayed_work irq_work;
	struct task_struct *irq_worker_task;
	struct wakeup_source *typec_wakelock;
	struct mutex		typec_mode_lock;
#if defined(__SGM7220_ADD_ADAPTER__)
	struct power_supply *charger_psy;
	const char *adapter_dev_name;
	struct adapter_device *adapter_dev;
#endif
#if defined(__SGM7220_ADD_PR_SWITCH__)
	bool				pr_swap_in_progress;
	int					dr_mode;
	struct mutex		typec_lock;
	struct delayed_work	role_reversal_check;
#endif
};

static struct sgm7220_info *sgm7220info = NULL;

/******************************************************************************
 * 
 * Register read and write
 * 
 ******************************************************************************/
/* i2c operate interfaces */
static int sgm7220_read_reg(struct i2c_client *i2c, u8 reg, u8 *dest)
{
	struct sgm7220_info *info = i2c_get_clientdata(i2c);
	int ret;

	mutex_lock(&info->mutex);
	ret = i2c_smbus_read_byte_data(i2c, reg);
	mutex_unlock(&info->mutex);
	if (ret < 0) {
		pr_err("%s: (0x%x) error, ret(%d)\n", __func__, reg, ret);
		return ret;
	}

	ret &= 0xff;
	*dest = ret;
	return 0;
}

static int sgm7220_write_reg(struct i2c_client *i2c, u8 reg, u8 value)
{
	struct sgm7220_info *info = i2c_get_clientdata(i2c);
	int ret;

	mutex_lock(&info->mutex);
	ret = i2c_smbus_write_byte_data(i2c, reg, value);
	mutex_unlock(&info->mutex);
	if (ret < 0)
		pr_err("%s: (0x%x) error, ret(%d)\n", __func__, reg, ret);

	return ret;
}

static int sgm7220_update_reg(struct i2c_client *i2c, u8 reg, u8 val, u8 mask)
{
	struct sgm7220_info *info = i2c_get_clientdata(i2c);
	int ret;
	u8 old_val, new_val;

	mutex_lock(&info->mutex);
	ret = i2c_smbus_read_byte_data(i2c, reg);

	if (ret >= 0) {
		old_val = ret & 0xff;
		new_val = (val & mask) | (old_val & (~mask));
		ret = i2c_smbus_write_byte_data(i2c, reg, new_val);
	}
	mutex_unlock(&info->mutex);
	return ret;
}

/******************************************************************************
 * 
 * create device nodes
 * 
 ******************************************************************************/
/*
static ssize_t mode_select_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	u8 value;
	int ret;

	ret = sgm7220_read_reg(info->i2c, REG_SET, &value);
	if (ret < 0) {
		pr_err("%s: read reg fail!\n", __func__);
		return ret;
	}
	value = (value & SET_MODE_SELECT) >> SET_MODE_SELECT_SHIFT;
	return snprintf(buf, PAGE_SIZE, "%d\n", value);
}

static ssize_t mode_select_store(struct device *dev,
				 struct device_attribute *attr, const char *buf, size_t size)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	int mode;
	u8 value;
	int ret;

	ret = kstrtoint(buf, 0, &mode);
	if (ret != 0)
		return -EINVAL;

	if (mode == MODE_DFP)
		value = SET_MODE_SELECT_SRC;
	else if (mode == MODE_UFP)
		value = SET_MODE_SELECT_SNK;
	else if (mode == MODE_DRP)
		value = SET_MODE_SELECT_DRP;
	else
		return -EINVAL;

	value = value << SET_MODE_SELECT_SHIFT;
	ret = sgm7220_update_reg(info->i2c, REG_SET, value, SET_MODE_SELECT);
	if (ret < 0) {
		pr_err("%s: update reg fail!\n", __func__);
	}
	return ret;
}

static ssize_t current_advertise_show(struct device *dev,
				      struct device_attribute *attr, char *buf)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	u8 value;
	int ret;

	ret = sgm7220_read_reg(info->i2c, REG_MOD, &value);
	if (ret < 0) {
		pr_err("%s: read reg fail!\n", __func__);
		return ret;
	}
	value = (value & MOD_CURRENT_MODE_ADVERTISE) >> MOD_CURRENT_MODE_ADVERTISE_SHIFT;
	return snprintf(buf, PAGE_SIZE, "%d\n", value);
}

static ssize_t current_advertise_store(struct device *dev,
				       struct device_attribute *attr, const char *buf, size_t size)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	int mode;
	u8 value;
	int ret;

	ret = kstrtoint(buf, 0, &mode);
	if (ret != 0)
		return -EINVAL;

	if (mode == HOST_CUR_USB)
		value = MOD_CURRENT_MODE_ADVERTISE_DEFAULT;
	else if (mode == HOST_CUR_1P5)
		value = MOD_CURRENT_MODE_ADVERTISE_MID;
	else if (mode == HOST_CUR_3A)
		value = MOD_CURRENT_MODE_ADVERTISE_HIGH;
	else
		return -EINVAL;

	value = value << MOD_CURRENT_MODE_ADVERTISE_SHIFT;

	ret = sgm7220_update_reg(info->i2c, REG_MOD, value, MOD_CURRENT_MODE_ADVERTISE);
	if (ret < 0) {
		pr_err("%s: update reg fail!\n", __func__);
	}
	return ret;
}

static ssize_t current_detect_show(struct device *dev,
				   struct device_attribute *attr, char *buf)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	u8 value;
	int ret;

	ret = sgm7220_read_reg(info->i2c, REG_MOD, &value);
	if (ret < 0) {
		pr_err("%s: read reg fail!\n", __func__);
		return ret;
	}
	value = (value & MOD_CURRENT_MODE_DETECT) >> MOD_CURRENT_MODE_DETECT_SHIFT;

	if (value == MOD_CURRENT_MODE_DETECT_DEFAULT)
		return snprintf(buf, PAGE_SIZE, "500mA or 900mA\n");
	else if (value == MOD_CURRENT_MODE_DETECT_MID)
		return snprintf(buf, PAGE_SIZE, "mid 1P5A\n");
	else if (value == MOD_CURRENT_MODE_DETECT_HIGH)
		return snprintf(buf, PAGE_SIZE, "high 3A\n");
	else if (value == MOD_CURRENT_MODE_DETECT_ACCESSARY)
		return snprintf(buf, PAGE_SIZE, "accessary 500mA\n");
	else
		return snprintf(buf, PAGE_SIZE, "unknown\n");
}

static ssize_t gpio_debug_show(struct device *dev,
			       struct device_attribute *attr, char *buf)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);

	return snprintf(buf, PAGE_SIZE, "EN_CC_MUX:%d\n",
			gpio_get_value(info->en_gpio[EN_CC_MUX]));
}

static ssize_t gpio_debug_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	int level;
	int ret;

	ret = kstrtoint(buf, 0, &level);
	if (ret != 0)
		return -EINVAL;

	if (info->en_gpio[EN_CC_MUX])
		gpio_direction_output(info->en_gpio[EN_CC_MUX], !!level);

	return 1;
}

static ssize_t show_reg_info(struct device *dev,
			      struct device_attribute *attr, char *buf)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	u8 reg_addr[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
					0x06, 0x07, 0x08, 0x09, 0x0A, 0x45};
	char reg_info[256] = { 0 };
	u8 reg_val;
	int i, ret;

	memset(reg_info, 0, 256);
	for (i = 0; i < sizeof(reg_addr) / sizeof(u8); i++) {
		reg_val = 0;
		ret = sgm7220_read_reg(info->i2c, reg_addr[i], &reg_val);
		if (ret < 0) {
			pr_err("%s: read reg[0x%x] fail!\n", __func__, reg_addr[i]);
			break;
		}
		snprintf(reg_info + i * 15, 256 - i * 15,
			"reg[0x%02x]=0x%02x\n", reg_addr[i], reg_val);
	}

	if (ret < 0) {
		pr_err("%s: read reg info fail!\n", __func__);
		return snprintf(buf, PAGE_SIZE, "read reg info: Error\n");
	} else {
		return snprintf(buf, PAGE_SIZE, reg_info);
	}
}

static ssize_t show_chip_info(struct device *dev,
			      struct device_attribute *attr, char *buf)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);
	u8 reg_val;
	char name[ID_REG_LEN + 1] = { 0 };
	int i, j, ret;

	for (i = 0, j = 0; i < ID_REG_LEN; i++) {
		reg_val = 0;
		ret = sgm7220_read_reg(info->i2c, (REG_BASE + (ID_REG_LEN - 1) - i), &reg_val);
		if (ret < 0) {
			pr_err("%s: read reg fail!\n", __func__);
			snprintf(name, ID_REG_LEN, "%s", "Error");
			break;
		}
		if (!reg_val) {
			j++;
			continue;
		}
		name[i - j] = (char)reg_val;
	}

	return snprintf(buf, PAGE_SIZE, "USB TypeC chip info:\n"
		"	Manufacture: TI\n"
		"	Chip Name  : %s\n",
		name);
}

#define TYPE_C_ATTR(field, format_string) \
		static ssize_t \
		field ## _show(struct device *dev, struct device_attribute *attr, \
			char *buf) \
		{ \
			struct sgm7220_info *info = dev_get_drvdata(dev); \
			return snprintf(buf, PAGE_SIZE, \
			format_string, info->type_c_param.field); \
		} \
		static DEVICE_ATTR(field, S_IRUGO, field ## _show, NULL);

static DEVICE_ATTR(gpio_debug,  S_IRUGO | S_IWUSR,
			gpio_debug_show, gpio_debug_store);
static DEVICE_ATTR(mode_select, S_IRUGO | S_IWUSR,
			mode_select_show, mode_select_store);
static DEVICE_ATTR(current_advertise, S_IRUGO | S_IWUSR,
			current_advertise_show, current_advertise_store);
static DEVICE_ATTR(current_det_string, S_IRUGO, current_detect_show, NULL);
static DEVICE_ATTR(reg_info, S_IRUGO, show_reg_info, NULL);
static DEVICE_ATTR(chip_info, S_IRUGO, show_chip_info, NULL);

TYPE_C_ATTR(current_det, "%d\n")
TYPE_C_ATTR(accessory_attach, "%d\n")
TYPE_C_ATTR(active_cable_attach, "%d\n")
TYPE_C_ATTR(attach_state, "%d\n")
TYPE_C_ATTR(cable_dir, "%d\n")

static struct device_attribute *usb_typec_attributes[] = {
	&dev_attr_mode_select,
	&dev_attr_current_advertise,
	&dev_attr_current_det_string,
	&dev_attr_chip_info,
	&dev_attr_reg_info,
	&dev_attr_current_det,
	&dev_attr_accessory_attach,
	&dev_attr_active_cable_attach,
	&dev_attr_attach_state,
	&dev_attr_cable_dir,
	&dev_attr_gpio_debug,
	NULL
};
*/

/******************************************************************************
 * 
 * Support adaptor class
 * 
 ******************************************************************************/
#if defined(__SGM7220_ADD_ADAPTER__)
static int sgm7220_notify_typec_init_done(struct sgm7220_info *info)
{
	int ret = 0;
	union power_supply_propval propval;

	pr_err("set init done!\n");
	/* Get chg type det power supply */
	if (!info->charger_psy) {
		info->charger_psy = power_supply_get_by_name("charger");
		if (!info->charger_psy) {
			pr_err("charger psy get fail\n");
			return -EINVAL;
		}
	}

	propval.intval = 1;
	ret = power_supply_set_property(info->charger_psy,
				POWER_SUPPLY_PROP_PRESENT, &propval);
	if (ret < 0)
		pr_err("Set psy present fail(%d)\n", ret);
	else
		pr_info("Typec init done!\n");

	return ret;
}

static int sgm7220_notify_typec_cc_changed(struct sgm7220_info *info, int notify)
{
	int ret = 0;
	union power_supply_propval propval;

	pr_err("set CC change is %d\n", notify);
	/* Get chg type det power supply */
	if (!info->charger_psy) {
		info->charger_psy = power_supply_get_by_name("charger");
		if (!info->charger_psy) {
			pr_err("charger psy get fail\n");
			return -EINVAL;
		}
	}

	propval.intval = notify;
	ret = power_supply_set_property(info->charger_psy,
				POWER_SUPPLY_PROP_USB_TYPE, &propval);
	if (ret < 0)
		pr_err("psy online fail(%d)\n", ret);
	else
		pr_info("Type CC mode = %d\n", notify);

	return ret;
}

static int sgm7220_get_prop_typec_cc_orientation(struct sgm7220_info *info) {
	int ret;
	u8 reg_val;
	u8 val;

	ret = sgm7220_read_reg(info->i2c, REG_INT, &reg_val);
	if (ret < 0) {
		pr_err("Couldn't read TYPE_C_STATUS_4 ret=%d\n", ret);
		return ret;
	}

	val = ((reg_val & INT_CABLE_DIR) >> INT_CABLE_DIR_SHIFT);

	return val;
}

static int sgm7220_get_prop_typec_rp_level(struct sgm7220_info *info) {
	u8 value;
	int ret;

	ret = sgm7220_read_reg(info->i2c, REG_MOD, &value);
	if (ret < 0) {
		pr_err("%s: read reg fail!\n", __func__);
		return ret;
	}
	value = (value & MOD_CURRENT_MODE_DETECT) >> MOD_CURRENT_MODE_DETECT_SHIFT;

	if (value == MOD_CURRENT_MODE_DETECT_DEFAULT)
		return 500;
	else if (value == MOD_CURRENT_MODE_DETECT_MID)
		return 1500;
	else if (value == MOD_CURRENT_MODE_DETECT_HIGH)
		return 3000;
	else if (value == MOD_CURRENT_MODE_DETECT_ACCESSARY)
		return 0;
	else
		return 0;
}

static int sgm7220_typec_get_property(struct adapter_device *dev, enum adapter_property sta)
{
	struct sgm7220_info *info;
	int ret;

	info = (struct sgm7220_info *)adapter_dev_get_drvdata(dev);
	if (info == NULL)
		return -1;

	switch (sta) {
	case TYPEC_RP_LEVEL:
		{
			ret = sgm7220_get_prop_typec_rp_level(info);
		}
		break;

	case TYPE_CC_MODE:
		{
			if (info->type_c_param.attach_state & SET_MODE_SELECT_DEFAULT)
				ret = 0;
			else if (info->type_c_param.attach_state & SET_MODE_SELECT_SRC)
				ret = 1;
			else if (info->type_c_param.attach_state & SET_MODE_SELECT_SNK)
				ret = 2;
			else
				ret = 3;
			break;
		}
		break;

	case TYPE_CC_ORIENTATION:
		{
			ret = sgm7220_get_prop_typec_cc_orientation(info);
		}
		break;

	default:
		{
			ret = -1;
		}
		break;
	}

	return ret;
}

static struct adapter_ops adapter_ops = {
	.get_property = sgm7220_typec_get_property,
};
#endif

/******************************************************************************
 * 
 * support switch ufp/dfp from Android OS
 * 
 ******************************************************************************/
#if defined(__SGM7220_ADD_PR_SWITCH__)
static int sgm7220_force_dr_mode(struct sgm7220_info *info, int mode)
{
	int ret;
	u8 value;

	if (mode == TYPEC_PORT_SRC)
		value = SET_MODE_SELECT_SRC;
	else if (mode == TYPEC_PORT_SNK)
		value = SET_MODE_SELECT_SNK;
	else if (mode == TYPEC_PORT_DRP)
		value = SET_MODE_SELECT_DRP;
	else
		return -EINVAL;

	value = value << SET_MODE_SELECT_SHIFT;
	if (mode == TYPEC_PORT_DRP) {
		ret = sgm7220_update_reg(info->i2c, REG_SET, value, SET_MODE_SELECT);
		if (ret < 0) {
			pr_err("write REG_SET DRP failed\n");
			return -EINVAL;
		}
		ret = sgm7220_update_reg(info->i2c, REG_CTL, 0x0, CTR_DISABLE_RD_RP);
		if (ret < 0) {
			pr_err("write REG_CTL DRP failed\n");
			return -EINVAL;
		}
	} else {
		ret = sgm7220_update_reg(info->i2c, REG_CTL, 0x4, CTR_DISABLE_RD_RP);
		if (ret < 0) {
			pr_err("write REG_CTL 1 failed\n");
			return -EINVAL;
		}
		msleep(200);
		ret = sgm7220_update_reg(info->i2c, REG_SET, value, SET_MODE_SELECT);
		if (ret < 0) {
			pr_err("write REG_SET failed\n");
			return -EINVAL;
		}
		ret = sgm7220_update_reg(info->i2c, REG_CTL, 0x0, CTR_DISABLE_RD_RP);
		if (ret < 0) {
			pr_err("write REG_CTL 0 failed\n");
			return -EINVAL;
		}
	}

	info->dr_mode = mode;

	return ret;
}

int sgm7220_typec_port_type_set(struct typec_port *port, enum typec_port_type type)
{
	struct sgm7220_info *info = typec_get_drvdata(port);
	int ret = 0;

	__pm_wakeup_event(info->typec_wakelock, 5000);
	mutex_lock(&info->typec_lock);
	if ((info->pr_swap_in_progress) || (type == TYPEC_PORT_DRP)) {
		pr_err("%s: ignoring port type request type = %d swap_in_progress = %d\n",
				__func__, type, info->pr_swap_in_progress);
		goto unlock;
	}
	pr_err("%s: requested role %s\n", __func__, type ? "SINK" : "SOURCE");

	info->pr_swap_in_progress = true;
	ret = sgm7220_force_dr_mode(info, type);
	if (ret < 0) {
		info->pr_swap_in_progress = false;
		pr_err("%s: update reg fail!\n", __func__);
		goto unlock;
	}

	/*
	 * As per the hardware requirements,
	 * schedule the work with required delay.
	 */
	if (!(delayed_work_pending(&info->role_reversal_check))) {
		cancel_delayed_work_sync(&info->role_reversal_check);
		schedule_delayed_work(&info->role_reversal_check,
			msecs_to_jiffies(SGM7220_ROLE_REVERSAL_DELAY_MS));
	}

unlock:
	mutex_unlock(&info->typec_lock);
	return ret;
}

static void sgm7220_typec_role_check_work(struct work_struct *work)
{
	struct sgm7220_info *info = container_of(work, struct sgm7220_info,
					role_reversal_check.work);
	int rc = 0;

	mutex_lock(&info->typec_lock);
	switch (info->dr_mode) {
	case TYPEC_PORT_SRC:
		if (!(info->type_c_param.attach_state & SET_MODE_SELECT_SNK)) {
			pr_err("%s: not latched to DFP in %d msecs. Reset to DRP!\n",
						__func__, SGM7220_ROLE_REVERSAL_DELAY_MS);
			rc = sgm7220_force_dr_mode(info, TYPEC_PORT_DRP);
			if (rc < 0)
				pr_err("%s: Failed to set DRP mode, rc=%d\n", __func__,  rc);
		}
		break;
	case TYPEC_PORT_SNK:
		if (!(info->type_c_param.attach_state & SET_MODE_SELECT_SRC)) {
			pr_err("%s: not latched to UFP in %d msecs. Reset to DRP!\n",
						__func__, SGM7220_ROLE_REVERSAL_DELAY_MS);
			rc = sgm7220_force_dr_mode(info, TYPEC_PORT_DRP);
			if (rc < 0)
				pr_err("%s: Failed to set DRP mode, rc=%d\n", __func__, rc);
		}
		break;
	default:
		pr_debug("Already in DRP mode\n");
		break;
	}

	info->pr_swap_in_progress = false;
	mutex_unlock(&info->typec_lock);
}

static const struct typec_operations sgm7220_typec_ops = {
	.port_type_set = sgm7220_typec_port_type_set,
};
#endif

/******************************************************************************
 * 
 * Process sink/src attach/detach
 * 
 ******************************************************************************/
static void process_src_sink_attach(struct sgm7220_info *info)
{
	struct typec_partner_desc desc;

	if (info->port) {
		if (info->type_c_param.attach_state & SET_MODE_SELECT_SNK) {
			typec_set_pwr_role(info->port, TYPEC_SOURCE);
			typec_set_data_role(info->port, TYPEC_HOST);
			/* Later USB host mode*/
		} else if (info->type_c_param.attach_state & SET_MODE_SELECT_SRC) {
			typec_set_pwr_role(info->port, TYPEC_SINK);
			typec_set_data_role(info->port, TYPEC_DEVICE);
		} else {
			pr_err("%s : Not sink or src!\n", __func__);
			return;
		}
		desc.usb_pd = 0;
		desc.accessory = TYPEC_ACCESSORY_NONE;
		desc.identity = NULL;
		info->partner = typec_register_partner(info->port, &desc);
		if (IS_ERR(info->partner)) {
				pr_err("%s : Type-C partner register failed!\n", __func__);
		}
	}
}

static void process_src_sink_detach(struct sgm7220_info *info)
{
	if (info->port) {
		typec_set_pwr_role(info->port, TYPEC_SINK);
		typec_set_data_role(info->port, TYPEC_DEVICE);
		if (!IS_ERR(info->partner))
			typec_unregister_partner(info->partner);
		info->partner = NULL;
	}
#if defined(__SGM7220_ADD_PR_SWITCH__)
	/* reset to DRP mode */
	if (!info->pr_swap_in_progress) {
		pr_err("Detach reset DRP\n");
		sgm7220_force_dr_mode(info, TYPEC_PORT_DRP);
		if (info->port)
			info->port->port_type = TYPEC_PORT_DRP;
	}
#endif
}

/* Detect non-DFP->DFP changes that happen more than 3 times within 10 Secs */
static void state_disorder_detect(struct sgm7220_info *info)
{
	unsigned long timeout;

	/* count the (non-DFP -> DFP) changes */
	if ((info->monitor.former_state != info->type_c_param.attach_state)
	    && (info->type_c_param.attach_state == CABLE_STATE_AS_DFP)) {
		if (!info->monitor.count) {
			info->monitor.time_before = jiffies;
		}
		info->monitor.count++;
	}

	/* store the state */
	info->monitor.former_state = info->type_c_param.attach_state;

	if (info->monitor.count > 3) {
		timeout = msecs_to_jiffies(10 * 1000); /* 10 Seconds */
		if (time_before(jiffies, info->monitor.time_before + timeout)) {
			info->monitor.err_detected = 1;
			/* disbale id irq before qpnp react to cc chip's id output */
			//interfere_id_irq_from_usb(0);
		}
		info->monitor.count = 0;
	}

	if ((info->type_c_param.attach_state == CABLE_STATE_NOT_ATTACHED)
	    && info->monitor.err_detected) {
		/* enable id irq */
		//interfere_id_irq_from_usb(1);
		info->monitor.err_detected = 0;
	}
}

static void process_mode_register(struct sgm7220_info *info, u8 status)
{
	u8 val;
	u8 tmp = status;

	/* check current_detect */
	val = ((tmp & MOD_CURRENT_MODE_DETECT) >> MOD_CURRENT_MODE_DETECT_SHIFT);
	info->type_c_param.current_det = val;

	/* check accessory attch */
	tmp = status;
	val = ((tmp & MOD_ACCESSORY_CONNECTED) >> MOD_ACCESSORY_CONNECTED_SHIFT);
	info->type_c_param.accessory_attach = val;

	/* check cable attach */
	tmp = status;
	val = (tmp & MOD_ACTIVE_CABLE_DETECTION);
}

static void process_interrupt_register(struct sgm7220_info *info, u8 status)
{
	u8 val;
	u8 tmp = status;

	/* check attach state */
	val = ((tmp & INT_ATTACHED_STATE) >> INT_ATTACHED_STATE_SHIFT);
	if (info->type_c_param.attach_state == val) {
		pr_err("%s : int sts repeat!\n", __func__);
		return;
	}
	info->type_c_param.attach_state = val;

	/* update current adv when act as DFP */
	if (info->type_c_param.attach_state == CABLE_STATE_AS_DFP ||
	    info->type_c_param.attach_state == CABLE_STATE_TO_ACCESSORY) {
		val = (HOST_CUR_USB << MOD_CURRENT_MODE_ADVERTISE_SHIFT);
	} else {
		val = (HOST_CUR_USB << MOD_CURRENT_MODE_ADVERTISE_SHIFT);
	}

	if (info->type_c_param.attach_state == SET_MODE_SELECT_DEFAULT) {
		process_src_sink_detach(info);
#if defined(__SGM7220_ADD_ADAPTER__)
		sgm7220_notify_typec_cc_changed(info, 0);
#endif
	} else {
		process_src_sink_attach(info);
#if defined(__SGM7220_ADD_ADAPTER__)
		if (info->type_c_param.attach_state & SET_MODE_SELECT_SRC)
			sgm7220_notify_typec_cc_changed(info, 1);
		else if (info->type_c_param.attach_state & SET_MODE_SELECT_SNK)
			sgm7220_notify_typec_cc_changed(info, 2);
		else
			sgm7220_notify_typec_cc_changed(info, 3);
#endif
	}

	pr_err("%s : type:%d\n", __func__, info->type_c_param.attach_state);

	sgm7220_update_reg(info->i2c, REG_MOD, val, MOD_CURRENT_MODE_ADVERTISE);

	/* in case configured as DRP may detect some non-standard SDP */
	/* chargers as UFP, which may lead to a cyclic switching of DFP */
	/* and UFP on state detection result. */
	state_disorder_detect(info);

	/* check cable dir */
	tmp = status;
	val = ((tmp & INT_CABLE_DIR) >> INT_CABLE_DIR_SHIFT);
	info->type_c_param.cable_dir = val;
}

static void sgm7220_irq_work_handler(struct kthread_work *work) {
	struct kthread_delayed_work *delay_thread_work = NULL;
	struct sgm7220_info *info = NULL;

	u8 modval;
	u8 intval;
	int ret;

	delay_thread_work = container_of(work, struct kthread_delayed_work, work);
	if(delay_thread_work == NULL) {
		pr_err("Cann't get sgm7220_irq_work\n");
		return;
	}
	info = container_of(delay_thread_work, struct sgm7220_info, irq_work);
	if(info == NULL) {
		pr_err("Cann't get info\n");
		return;
	}

	mutex_lock(&info->typec_mode_lock);
	/* Reset INT port */
	ret = sgm7220_read_reg(info->i2c, REG_INT, &intval);
	if (ret) {
		pr_err("%s REG_INT read failed!\n");
		goto done;
	}

	ret = sgm7220_read_reg(info->i2c, REG_MOD, &modval);
	if (ret) {
		pr_err("%s REG_MOD read failed!\n");
		goto done;
	}

	pr_err("%s mode_register:0x%x\n", __func__, modval);
	process_mode_register(info, modval);
	pr_err("%s interrupt_register:0x%x\n", __func__, intval);
	process_interrupt_register(info, intval);

done:
	ret = sgm7220_update_reg(info->i2c,
				   REG_INT, (0x1 << INT_INTERRUPT_STATUS_SHIFT), INT_INTERRUPT_STATUS);
	if (ret < 0) {
		pr_err("%s: update reg fail!\n", __func__);
	}

	if (info->dev->parent)
		pm_runtime_put(info->dev->parent);
	mutex_unlock(&info->typec_mode_lock);
}

static irqreturn_t sgm7220_irq_thread(int irq, void *handle)
{
	struct sgm7220_info *info = (struct sgm7220_info *)handle;

	pr_err("%s enter\n", __func__);
	__pm_wakeup_event(info->typec_wakelock, 500);
	/* Runtime resume the i2c bus device from runtime suspend */
	if (info->dev->parent)
		pm_runtime_get(info->dev->parent);

	if (info->irq_worker_task)
		kthread_queue_delayed_work(&info->irq_worker, &info->irq_work,
					msecs_to_jiffies(50));

	return IRQ_HANDLED;
}

/******************************************************************************
 * 
 * device suspend/resume
 * 
 ******************************************************************************/
#ifdef TYPEC_PM_OP
static int sgm7220_suspend(struct device *dev)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);

	pr_info("%s enter\n", __func__);
	disable_irq_wake(info->i2c->irq);
	disable_irq(info->i2c->irq);

	if (info->en_gpio[EN_CC_MUX])
		gpio_direction_output(info->en_gpio[EN_CC_MUX], 0);

	return 0;
}

static int sgm7220_resume(struct device *dev)
{
	struct sgm7220_info *info = dev_get_drvdata(dev);

	pr_info("%s enter\n", __func__);

	if (info->en_gpio[EN_CC_MUX])
		gpio_direction_output(info->en_gpio[EN_CC_MUX], 1);

	msleep(20);

	enable_irq_wake(info->i2c->irq);
	enable_irq(info->i2c->irq);

	return 0;
}
#endif

/******************************************************************************
 * 
 * device initialization
 * 
 ******************************************************************************/
static int sgm7220_initialization(struct sgm7220_info *info)
{
	int ret = 0;
	u8 reg_val;
	char name[ID_REG_LEN + 1] = { 0 };
	int i, j;
	/* do initialization here, before enable irq,
	 * clear irq,
	 * config DRP/UFP/DFP mode,
	 * and etc..
	 */
	pr_err("%s enter\n", __func__);

	for (i = 0, j = 0; i < ID_REG_LEN; i++) {
		reg_val = 0;
		ret = sgm7220_read_reg(info->i2c, (REG_BASE + (ID_REG_LEN - 1) - i), &reg_val);
		if (ret < 0) {
			pr_err("%s: read reg fail!\n", __func__);
			snprintf(name, ID_REG_LEN, "%s", "Error");
			break;
		}
		if (!reg_val) {
			j++;
			continue;
		}
		name[i - j] = (char)reg_val;
	}
	if (strcmp(name,SGM7220_ID) != 0) {
		pr_err("%s: error name:%s\n", __func__, name);
		return -1;
	}

	/* 1 Disable term(default 0) [bit 0]
	 * 2 Source pref(DRP performs try.SNK 01) [bit 2:1]
	 * 3 Soft reset(default 0) [bit 3]
	 * 4 Mode slect(DRP start from Try.SNK 00) [bit 5:4]
	 * 5 Debounce time of voltage change on CC pin(default 0) [bit 7:6]
	 */
	reg_val = 0x30;
	ret = sgm7220_write_reg(info->i2c, REG_SET, reg_val);
	if (ret < 0) {
		pr_err("%s: init REG_SET fail!\n", __func__);
		return ret;
	}

	ret = sgm7220_update_reg(info->i2c, REG_INT, 0x1, 0x1);
	if (ret < 0) {
		pr_err("%s: Disable UFP access fail!\n", __func__);
		return ret;
	}

	/* CURRENT MODE ADVERTISE 3A [bit 7:6] */
	reg_val = (HOST_CUR_USB << MOD_CURRENT_MODE_ADVERTISE_SHIFT);
	ret = sgm7220_update_reg(info->i2c, REG_MOD, reg_val, MOD_CURRENT_MODE_ADVERTISE);
	if (ret < 0) {
		pr_err("%s: init REG_MOD fail!\n", __func__);
		return ret;
	}

	return ret;
}

/******************************************************************************
 * 
 * create proc file for typec cc dir
 * 
 ******************************************************************************/
static int cc_dir_read(struct seq_file *m, void *v)
{
	int rc = 0;
	int ret = 0;
	u8 reg_val;
	u8 temp;

	ret = sgm7220_read_reg(sgm7220info->i2c, REG_INT, &reg_val);
	if (ret) {
		pr_err("%s: failed to read\n", __func__);
		return ret;
	}

	temp = (reg_val & INT_ATTACHED_STATE) >> INT_ATTACHED_STATE_SHIFT;

	if (temp) {
		rc = ((reg_val & INT_CABLE_DIR) >> INT_CABLE_DIR_SHIFT) + 1;
	} else {
		rc = 0;
	}
	pr_err("%s: reg_val=%d\n", __func__, rc);

	/* 00: Standby;
	 * 01: CC1 connected;
	 * 10: CC2 connected;
	 * 11: CC1 and CC2 connected;
	 */
	seq_printf(m, "%d\n", rc);
	pr_err("%s: cc_dir=%d\n", __func__,rc);
	return 0;
}

static int cc_dir_open(struct inode *inode, struct file *file)
{
	return single_open(file, cc_dir_read, NULL);
}

static const struct proc_ops cc_dir_proc_fops = {
	.proc_open = cc_dir_open,
	.proc_read = seq_read,
};

static int sgm7220_create_proc_info(struct device *cdev)
{
	struct proc_dir_entry *pe;
	struct proc_dir_entry *proc_dir = NULL;

	proc_dir = proc_mkdir("type_cc", NULL);
	if (!proc_dir) {
		dev_err(cdev, "[%s]: mkdir /proc/type_cc failed\n", __func__);
		return -ENOMEM;
	}

	pe = proc_create("cc_dir", S_IRUGO | S_IWUSR, proc_dir, &cc_dir_proc_fops);
	if (!pe) {
		dev_err(cdev, "[%s]: mkdir /proc/type_cc/cc_dir failed\n", __func__);
		return -ENOMEM;
	}

	return 0;
}

/******************************************************************************
 * 
 * parse dts config
 * 
 ******************************************************************************/
static int sgm7220_parse_gpio_from_dts(struct device_node *np, struct sgm7220_info *info)
{
	int ret;
	int i;
	const char *pinctrl_state_name;
	struct pinctrl *irq_pinctrl;

	pinctrl_state_name = of_get_property(np, "pinctrl-names", NULL);
	/* configure irq_pinctrl to enable irqs */
	if (pinctrl_state_name) {
		irq_pinctrl = pinctrl_get_select(&info->i2c->dev, pinctrl_state_name);
		if (IS_ERR(irq_pinctrl)) {
			pr_err("Could not get/set %s pinctrl state rc = %ld\n",
						pinctrl_state_name,
						PTR_ERR(irq_pinctrl));
			return PTR_ERR(irq_pinctrl);
		}
	}

	/* irq_line */
	ret = of_get_named_gpio(np, "sgm7220,irq_gpio", 0);
	if (ret < 0) {
		pr_err("%s: error invalid irq gpio err: %d\n", __func__, ret);
		return ret;
	}
	info->irq_gpio = ret;
	pr_err("%s: valid irq_gpio number: %d\n", __func__, ret);
	ret = gpio_request(info->irq_gpio, "sgm7220 irq");
	if (ret) {
		pr_err("%s: gpio %d request failed: %d\n", __func__, info->irq_gpio);
		return ret;
	}
	gpio_direction_input(info->irq_gpio);

	/*
	 * en gpios, for changing state during suspend to reduce power consumption,
	 * as these gpios may already have initial config in the dts files,
	 * getting gpio failure here may not be a critical error.
	 */
	for (i = 0; i < EN_GPIO_MAX; i++) {
		ret = of_get_named_gpio(np, en_gpio_info[i].name, 0);
		if (ret < 0) {
			info->en_gpio[en_gpio_info[i].index] = 0;
			pr_err("%s: error invalid en gpio [%s] err: %d\n",
			       __func__, en_gpio_info[i].name, ret);
		} else {
			info->en_gpio[en_gpio_info[i].index] = ret;
			pr_err("%s: valid en gpio number: %d\n", __func__, ret);
		}
	}

	return 0;
}

/******************************************************************************
 * 
 * probe
 * 
 ******************************************************************************/
static int sgm7220_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct sgm7220_info *info;
	struct device_node *np = client->dev.of_node;
	int ret, irq;
	struct sched_param param = { .sched_priority = MAX_RT_PRIO - 1 };

	info = kzalloc(sizeof(struct sgm7220_info), GFP_KERNEL);
	info->i2c = client;
	info->dev = &client->dev;
	pr_err("sgm7220 usb type-c ship start probe\n");

	i2c_set_clientdata(client, info);
	mutex_init(&info->mutex);
	mutex_init(&info->typec_mode_lock);
#if defined(__SGM7220_ADD_PR_SWITCH__)
	mutex_init(&info->typec_lock);
	INIT_DELAYED_WORK(&info->role_reversal_check, sgm7220_typec_role_check_work);
	info->dr_mode = TYPEC_PORT_DRP;
#endif
	/* register wakelock and init typec mode change work */
	info->typec_wakelock = wakeup_source_register(NULL, "typec suspend wakelock");
	if(!info->typec_wakelock)
		pr_err("%s: request wakelock error.\n", __func__);

	/* try initialize device before request irq */
	ret = sgm7220_initialization(info);
	if (ret < 0) {
		pr_err("%s: fails to do initialization %d\n", __func__, ret);
		goto err_init_dev;
	}

	kthread_init_worker(&info->irq_worker);
	info->irq_worker_task = kthread_run(kthread_worker_fn,
				&info->irq_worker, "%s", 
				"sgm7220 irq worker");
	if (IS_ERR(info->irq_worker_task)) {
		pr_err("%s: Could not create tcpc task\n", __func__);
	} else {
		sched_setscheduler(info->irq_worker_task, SCHED_FIFO, &param);
		kthread_init_delayed_work(&info->irq_work, sgm7220_irq_work_handler);
	}

	/* get gpio(s) */
	ret = sgm7220_parse_gpio_from_dts(np, info);
	if (ret < 0) {
		pr_err("%s: error invalid irq gpio number: %d\n", __func__, ret);
		goto err_pinctrl;
	}
	irq = gpio_to_irq(info->irq_gpio);
	if (irq < 0) {
		pr_err("%s: error gpio_to_irq returned %d\n", __func__, irq);
		goto err_request_irq;
	} else {
		pr_err("%s: requesting IRQ %d\n", __func__, irq);
		client->irq = irq;
	}

//	ret = request_threaded_irq(client->irq, NULL, sgm7220_irq_thread,
//				   IRQF_TRIGGER_FALLING | IRQF_ONESHOT, "sgm7220_irq", info);
	ret = request_irq(client->irq, sgm7220_irq_thread,
				IRQF_TRIGGER_FALLING | IRQF_NO_THREAD | IRQF_NO_SUSPEND,
				"sgm7220_irq", info);
	if (ret) {
		pr_err("%s: error failed to request IRQ\n", __func__);
		goto err_request_irq;
	}

	ret = enable_irq_wake(client->irq);
	if (ret < 0) {
		pr_err("%s: failed to enable wakeup src %d\n", __func__, ret);
		goto err_enable_irq;
	}

#if defined(__SGM7220_ADD_ADAPTER__)
    info->adapter_dev_name = "typec_adapter";
	info->adapter_dev = adapter_device_register(info->adapter_dev_name,
		info->dev, info, &adapter_ops, NULL);
	if (IS_ERR_OR_NULL(info->adapter_dev)) {
		pr_err("Register typec_adapter failed\n");
		ret = PTR_ERR(info->adapter_dev);
		goto err_enable_irq;
	} else {
		pr_err("Register typec_adapter successful\n");
	}

	adapter_dev_set_drvdata(info->adapter_dev, info);
#endif

	ret = sgm7220_create_proc_info(&client->dev);
	if (ret < 0) {
		pr_err("%s: could not create proc info\n", __func__);
		goto err_enable_irq;
	}

	info->typec_cap.revision = USB_TYPEC_REV_1_2;
	info->typec_cap.driver_data = info;
	info->typec_cap.prefer_role = TYPEC_NO_PREFERRED_ROLE;
	info->typec_cap.type = TYPEC_PORT_DRP;
	info->typec_cap.data = TYPEC_PORT_DRD;
#if defined(__SGM7220_ADD_PR_SWITCH__)
	info->typec_cap.ops = &sgm7220_typec_ops;
#endif
	info->port = typec_register_port(&client->dev, &info->typec_cap);
	if (IS_ERR(info->port)) {
		pr_err("%s: register type-c port failed!\n", __func__);
		info->port = NULL;
	}

	sgm7220info = info;
	sgm7220_irq_thread(0, info);
#if defined(__SGM7220_ADD_ADAPTER__)
	sgm7220_notify_typec_init_done(info);
#endif
	/* update typec info  */
	memset(typec_info, 0, sizeof(typec_info));
	memcpy(typec_info, SGM7220_NAME,
		strlen(SGM7220_NAME) > (sizeof(typec_info) - 1) ?
		(sizeof(typec_info) - 1) : strlen(SGM7220_NAME));

	pr_err("sgm7220 usb type-c chip finish probe\n");
	return 0;

err_enable_irq:
	free_irq(client->irq, NULL);
#if defined(__SGM7220_ADD_ADAPTER__)
	adapter_device_unregister(info->adapter_dev);
#endif
err_request_irq:
err_pinctrl:
err_init_dev:
#if defined(__SGM7220_ADD_PR_SWITCH__)
	cancel_delayed_work_sync(&info->role_reversal_check);
	mutex_destroy(&info->typec_lock);
#endif
	mutex_destroy(&info->mutex);
	mutex_destroy(&info->typec_mode_lock);
	i2c_set_clientdata(client, NULL);
	kfree(info);
	info = NULL;
	sgm7220info = NULL;
	return ret;
}

static int sgm7220_remove(struct i2c_client *client)
{
	struct sgm7220_info *info = i2c_get_clientdata(client);

	if (client->irq) {
		disable_irq_wake(client->irq);
		free_irq(client->irq, info);
	}

	mutex_destroy(&info->mutex);
	mutex_destroy(&info->typec_mode_lock);
	if (info->typec_wakelock)
		wakeup_source_unregister(info->typec_wakelock);
#if defined(__WUSB3801_ADD_PR_SWITCH__)
	mutex_destroy(&info->typec_lock);
	cancel_delayed_work_sync(&chg->role_reversal_check);
#endif /*__WUSB3801_ADD_PR_SWITCH__*/
	i2c_set_clientdata(client, NULL);

	kfree(info);
	return 0;
}

static void sgm7220_shutdown(struct i2c_client *client)
{
	struct sgm7220_info *info = i2c_get_clientdata(client);
	int ret;
	u8 value;

	if(client->irq)
		disable_irq(client->irq);
	kthread_cancel_delayed_work_sync(&info->irq_work);

	value = SET_MODE_SELECT_SNK;
	value = value << SET_MODE_SELECT_SHIFT;
	ret = sgm7220_update_reg(info->i2c, REG_SET, value, SET_MODE_SELECT);
	if (ret < 0) {
		pr_err("%s: update reg fail!\n", __func__);
	}

	/* reset the INT */
	ret = sgm7220_update_reg(info->i2c,
				   REG_INT, (0x1 << INT_INTERRUPT_STATUS_SHIFT), INT_INTERRUPT_STATUS);
	if (ret < 0) {
		pr_err("%s: update reg fail!\n", __func__);
	}
}

static const struct of_device_id sgm7220_dt_match[] = {
	{
		.compatible = "sgm,sgm7220",
	},
	{ },
};
MODULE_DEVICE_TABLE(of, sgm7220_dt_match);

static const struct i2c_device_id sgm7220_id_table[] = {
	{
		.name = "sgm7220",
	},
};

#ifdef TYPEC_PM_OP
static SIMPLE_DEV_PM_OPS(sgm7220_dev_pm, sgm7220_suspend, sgm7220_resume);
#endif

static struct i2c_driver sgm7220_i2c_driver = {
	.driver = {
		.name = "sgm7220",
		.of_match_table = of_match_ptr(sgm7220_dt_match),
#ifdef TYPEC_PM_OP
		.pm = &sgm7220_dev_pm,
#endif
	},
	.probe    = sgm7220_probe,
	.remove   = sgm7220_remove,
	.shutdown = sgm7220_shutdown,
	.id_table = sgm7220_id_table,
};

static __init int sgm7220_i2c_init(void)
{
	return i2c_add_driver(&sgm7220_i2c_driver);
}

static __exit void sgm7220_i2c_exit(void)
{
	i2c_del_driver(&sgm7220_i2c_driver);
}

module_init(sgm7220_i2c_init);
module_exit(sgm7220_i2c_exit);

MODULE_DESCRIPTION("I2C bus driver for sgm7220 USB Type-C");
MODULE_LICENSE("GPL v2");
